# lib/dbld2img.tcl
#
# Bibliotheksmodul für die Grafikerzeugung aus den Druckbildwerten
#
# Die interpolierten Bilder werden in Hintergrundthreads erstellt.
# Die Bilderzeugung wird mit der Prozedur startJPEG angeworfen,
# der eine Callback-Prozedur für die Fertigmeldung übergeben wird. (s. unten)
# Die JPEG-Daten sind dann mit 
#   ::tsv::get images $imagetype
# abrufbar.

package require Thread; # Debian tclthread

namespace eval ::DBLD2IMG {
    variable jpegoptions; # dict
    # Anzahl der Interpolationsthreads (nicht konfigurierbar:
    #   Mehr Threads führen nur zu mehr Verzögerung, verhindern aber keine Überlastung.)
    variable N_THREADS 5
    variable threads; # array
    variable busylevels; # array
    variable overload_handler_procnames [list]

    # Defaultoptions für JPEG
    if {[info exists ::JPEGOPTIONS]} {
        set jpegoptions [dict create {*}$::JPEGOPTIONS]
    } else {
        set jpegoptions [dict create -quality 60 -res 12 -grid no]
    }

    if {[info exists ::BGCOLOR]} {
        set bgcolor $::BGCOLOR
    } else {
        set bgcolor #404040
    }
    #TODO Der Aufruf von druckbild::photo_configure bgcolor $bgcolor
    # scheint auch außerhalb eines Threads auf alle Threads zu wirken,
    # weil der Wert in einer static - Variablen gespeichert wird.
    # => prüfen und dan hier ausbauen.

    # druckbild::jpegdata spiegelt das Druckbild horizontal
    # Dadurch ensteht unbeabsichtigt das gewünschte Ergebnis.
    #TODO Kommando überprüfen, so daß die Spiegelung bewußt herbeigeführt wird.
    for {set i 0} {$i < $N_THREADS} {incr i} {; #{{{
        set threads($i) [::thread::create "
            load $::SODIR/libtcl${tcl_version}vmkstationd.so
	
	        druckbild::photo_configure bgcolor $bgcolor
	
	        # Diese Prozedur wird von startJPEG benutzt.
	        # @param sender_id  ThreadId des Absenders (Thread der startJPEG aufgerufen hat)
	        # @param cbproc     Name der Callbackprozedur, mit der der Absender das erzeugte Bild übernimmt
            #                   Der Prozeduraufruf erfolgt mit dem nachfolgenden Parameter imagetype.
            # @param imagetype  Bildtyp: Elementname im Array images im shared Memory.
            # @param jpegoptions Die JPEG-Optionen als key/value Liste
	        # @param druckbild  Druckwerte zeilenweise von links oben nach rechts unten
	        # @param n_rows     Anzahl Zeilen
	        # @param n_cols     Anzahl Spalten
	        proc dbld2jpeg {sender_id cbproc imagetype jpegoptions druckbild n_rows n_cols} {
                if {\[catch {
	                set jpegimage \[druckbild::jpegdata {*}\$jpegoptions \$druckbild \$n_rows \$n_cols\]
	                ::tsv::set images \$imagetype \$jpegimage
	                ::thread::send -async \$sender_id \"\$cbproc \$imagetype\"
                } msg\]} {
	                ::thread::send -async \$sender_id \"::DBLD2IMG::logJPEG Error \$imagetype {\$msg}\"
                }
                # Thread auch im Fehlerfall wieder freigeben
                ::thread::send \$sender_id \"incr ::DBLD2IMG::busylevels($i) -1\"
	        }
	        thread::wait
	    "]
        set busylevels($i) 0
	    #}}}
    }

    # Fehlermeldung von einem JPEG-Thread ins Log schreiben
    proc logJPEG {level imagetype msg} {
        srvLog [namespace current] $level "${imagetype}: $msg"
    }

    # Grafikerzeugung in Hintergrundthread starten
    # @param cbproc     Name der Callbackprozedur, mit der der Absender das erzeugte Bild übernimmt
    #                   Der Prozeduraufruf erfolgt mit dem übergebenen Parameter imagetype.
    #                   Das fertige JPEG-Bild befindet sich dann im Element "$imagetype" der tsv "images".
    # @param druckbild  Druckwerte zeilenweise von links oben nach rechts unten
    # @param n_rows     Anzahl Zeilen
    # @param n_cols     Anzahl Spalten
    # @param imagetype  Bildtyp: Elementname im Array images im shared Memory.
    #                   TODO imagetype sollte Pflicht werden und
    # @param alt_jpegoptions key/value Liste abweichender JPEG-Optionen
    # @return   1 gestartet, 0 Überlast
    proc startJPEG {cbproc druckbild n_rows n_cols {imagetype jpegimage} {alt_jpegoptions {} }} {; #{{{
        variable N_THREADS
        variable threads
        variable busylevels
        variable jpegoptions
        variable overload_handler_procnames

        for {set i 0} {$i < $N_THREADS} {incr i} {
            if {$busylevels($i) == 0} {
                # Die alternativen Optionen in eine Kopie einbauen und so mit übergeben
                set opts [dict merge $jpegoptions $alt_jpegoptions]
                ::thread::send -async $threads($i) "dbld2jpeg [::thread::id] $cbproc $imagetype \"$opts\" \"$druckbild\" $n_rows $n_cols"
                incr busylevels($i)
                return 1
            }
        }
        srvLog [namespace current] Warn "startJPEG: Threads too busy for $imagetype."
        # Meldung an alle overloadHandler
        foreach overload_handler_procname $overload_handler_procnames {
            $overload_handler_procname $imagetype
        }
        return 0
        #}}}
    }; # proc startJPEG 


    # Globale JPEG-Optionen ändern
    # @param args   key/value Paare:
    #                   colorcontrast 0...7
    #                   bgcolor #%2x%2x%2x
    proc setGlobalJPEG {args} {; #{{{
        # Wieso funktioniert das?
        #   Die Bilder werden in einem eigenen Interpreter eines Thread generiert,
        #   der vorher sein eigenes load für die shared library ausgeführt hat.
        #   Ohne dieses load sind die Bibliothekskommandos nicht bekannt => Fehler.
        #   Die Konfigurationsvariablen sind im C-Kode static Variable.
        #   Wahrscheinlich gibt es die auch nach mehrfachem load in verschiedene Interpreter nur einmal.
        foreach {key value} $args {
            ::druckbild::photo_configure $key $value
        }
        #}}}
    }; # proc setGlobalJPEG 


    # (Weiteren) Handler für Überlastung hinzufügen
    # Gemeldet wird jede Überlastung.
    # Der muß den Bildtyp als Argument entgegennehmen.
    # @param cb_overload_handler    Name der Callback-Prozedur
    proc addOverloadHandler {cb_overload_handler} {; #{{{
        variable overload_handler_procnames

        # Vorsichtshalber prüfen, ob es den schon gibt
        if {[lsearch $overload_handler_procnames $cb_overload_handler] < 0} {
            lappend overload_handler_procnames $cb_overload_handler
            srvLog "[namespace current]::addOverloadHandler" Debug "$cb_overload_handler hinzugefügt"
        }
        #}}}
    }; #proc addOverloadHandler 


    # Handler für Überlastung entfernen
    # @param cb_overload_handler    Name der Callback-Prozedur
    proc removeOverloadHandler {cb_overload_handler} {; #{{{
        variable overload_handler_procnames

        set i [lsearch $overload_handler_procnames $cb_overload_handler]
        if {$i >=0} {
            set overload_handler_procnames [lreplace $overload_handler_procnames $i $i]
            srvLog [namespace current] Debug "::removeOverloadHandler $cb_overload_handler entfernt"
        } else {
            srvLog [namespace current] Warn "::removeOverloadHandler $cb_overload_handler nicht gefunden"
        }
        #}}}
    }; # proc removeOverloadHandler 


}; # namespace eval ::DBLD2IMG


