const script_host = location.origin;
const script_ws = script_host.replace('http', 'ws');

const app_dek = "dekubitus";

let ws;
try {
    ws = new WebSocket(`${script_ws}/apps/${app_dek}`, ["soap", "wamp"]);
} catch (error) {
    console.error('WebSocket-Fehler:', error);
}

// Initialisierung des Canvas
const canvas = document.getElementById('gameCanvas');
const ctx = canvas.getContext('2d');

// Spieler-Position und Eigenschaften
let playerX = canvas.width / 2;
const playerY = canvas.height - 50;
const playerWidth = 100;
const playerHeight = 20;

// Ball-Eigenschaften
let ballX = canvas.width / 2;
let ballY = canvas.height / 2;
const ballRadius = 10;
let ballSpeedX = 3;
let ballSpeedY = -3;

// Score
let score = 0;

// Blöcke
const blockRowCount = 4;
const blockColumnCount = 8;
const blockWidth = 60;
const blockHeight = 20;
const blockPadding = 10;
const blockOffsetTop = 30;
const blockOffsetLeft = 35;

// Array für die Blöcke
let blocks = [];
for (let r = 0; r < blockRowCount; r++) {
    blocks[r] = [];
    for (let c = 0; c < blockColumnCount; c++) {
        // Zufällige Platzierung (50% Chance, dass ein Block existiert)
        if (Math.random() > 0.5) {
            blocks[r][c] = { x: 0, y: 0, status: 1 }; // Status 1 bedeutet, der Block ist intakt
        } else {
            blocks[r][c] = { x: 0, y: 0, status: 0 }; // Kein Block
        }
    }
}

if (ws) {
    ws.onopen = () => {
        console.log('WebSocket verbunden');
    };

    ws.onmessage = (event) => {
        const data = JSON.parse(event.data);
    
        if (data.wsevent === 'normdata' && data.imagetype === 'dkb_live') {
            const values = data.values;
            const nCols = data.n_cols;
    
            // Anzahl der Zeilen berechnen
            const nRows = Math.floor(values.length / nCols);
    
            // Druck auf der linken und rechten Seite berechnen
            let leftPressure = 0;
            let rightPressure = 0;
    
            for (let row = 0; row < nRows; row++) {
                for (let col = 0; col < nCols; col++) {
                    const index = row * nCols + col;
                    if (col < nCols / 2) {
                        leftPressure += values[index]; // Linke Hälfte
                    } else {
                        rightPressure += values[index]; // Rechte Hälfte
                    }
                }
            }
    
            // Bewegung basierend auf der Druckverteilung
            const threshold = 10; // Empfindlichkeit, ab wann Bewegung ausgelöst wird
            if (leftPressure > rightPressure + threshold) {
                playerX -= 15; // Bewegung nach links
            } else if (rightPressure > leftPressure + threshold) {
                playerX += 15; // Bewegung nach rechts
            }
    
            // Begrenzung des Spielers im Canvas
            playerX = Math.max(0, Math.min(playerX, canvas.width - playerWidth));
        }
    };

    ws.onerror = (error) => {
        console.error('WebSocket-Fehler:', error);
    };

    ws.onclose = () => {
        console.log('WebSocket geschlossen');
    };
}

// Ball-Bewegung und Kollisionslogik
function updateBall() {
    ballX += ballSpeedX;
    ballY += ballSpeedY;

    // Kollision mit den Wänden
    if (ballX - ballRadius < 0 || ballX + ballRadius > canvas.width) {
        ballSpeedX = -ballSpeedX; // Richtung umkehren
    }

    if (ballY - ballRadius < 0) {
        ballSpeedY = -ballSpeedY; // Richtung umkehren
    }

    // Kollision mit dem Spieler
    if (
        ballY + ballRadius > playerY &&
        ballX > playerX &&
        ballX < playerX + playerWidth
    ) {
        ballSpeedY = -ballSpeedY; // Richtung umkehren
    }

    // Kollision mit den Blöcken
    for (let r = 0; r < blockRowCount; r++) {
        for (let c = 0; c < blockColumnCount; c++) {
            const block = blocks[r][c];
            if (block.status === 1) {
                if (
                    ballX > block.x &&
                    ballX < block.x + blockWidth &&
                    ballY > block.y &&
                    ballY < block.y + blockHeight
                ) {
                    ballSpeedY = -ballSpeedY; // Ball abprallen lassen
                    block.status = 0; // Block zerstören
                    score++; // Score erhöhen
                }
            }
        }
    }

    // Ball fällt runter
    if (ballY - ballRadius > canvas.height) {
        alert(`Game Over! Dein Score: ${score}`);
        resetGame();
    }
}

// Spiel zurücksetzen
function resetGame() {
    ballX = canvas.width / 2;
    ballY = canvas.height / 2;
    ballSpeedX = 3;
    ballSpeedY = -3;
    score = 0;

    // Blöcke neu initialisieren
    for (let r = 0; r < blockRowCount; r++) {
        for (let c = 0; c < blockColumnCount; c++) {
            blocks[r][c].status = Math.random() > 0.5 ? 1 : 0;
        }
    }
}

// Blöcke zeichnen
function drawBlocks() {
    for (let r = 0; r < blockRowCount; r++) {
        for (let c = 0; c < blockColumnCount; c++) {
            const block = blocks[r][c];
            if (block.status === 1) {
                const blockX = c * (blockWidth + blockPadding) + blockOffsetLeft;
                const blockY = r * (blockHeight + blockPadding) + blockOffsetTop;
                block.x = blockX;
                block.y = blockY;

                ctx.fillStyle = 'green';
                ctx.fillRect(blockX, blockY, blockWidth, blockHeight);
            }
        }
    }
}

// Spiellogik
function draw() {
    // Canvas löschen
    ctx.clearRect(0, 0, canvas.width, canvas.height);

    // Spieler zeichnen
    ctx.fillStyle = 'blue';
    ctx.fillRect(playerX, playerY, playerWidth, playerHeight);

    // Ball zeichnen
    ctx.beginPath();
    ctx.arc(ballX, ballY, ballRadius, 0, Math.PI * 2);
    ctx.fillStyle = 'red';
    ctx.fill();
    ctx.closePath();

    // Blöcke zeichnen
    drawBlocks();

    // Score anzeigen
    ctx.fillStyle = 'black';
    ctx.font = '20px Arial';
    ctx.fillText(`Score: ${score}`, 10, 20);

    // Ball-Bewegung aktualisieren
    updateBall();

    // Nächsten Frame anfordern
    requestAnimationFrame(draw);
}

draw();
