#!/usr/bin/env tclsh
#
# create_vlbsettings_sqlite3.tcl
#
# Datenbank für die Apps des Stationsdaemon erzeugen und füllen
# Nur für die Erstellung des Setups bei Velometrik
#

# Historie:
# 12.01.2023 Siegmar Müller Erstellt
#


package require tdbc
package require tdbc::sqlite3
package require tdbc::postgres


# Interessierende Datenbanken
set PG_CONNDATA [dict create \
    localhost {-host pc-32.local -user yadmin -password {} -port 5434 -db vmkprodukte_dev} \
    kronos {-host kronos -user yadmin -password {} -port 5433 -db vmkprodukte} \
    dev {-host db-3-1.velometrik.eu -user yadmin -password {} -port 5433 -db vmkprodukte_dev} \
    produktion {-host db-3-1.velometrik.de -user yadmin -password {} -port 5433 -db vmkprodukte} \
]
set PG_CONN_DEFAULT dev

set root_path "[file join {*}[lrange [file split [exec pwd]] 0 end-1]]"
puts "root_path = $root_path"
set dir [lindex [file split [exec pwd]] end]
puts "dir = $dir"
if {![file isdirectory "$root_path/custom/apps"] || "$dir" != "dev"} {
    puts "Das Skript muß im Unterverzeichnis dev des vmkstationd ausgeführt werden."
    exit 1
}

# Optionen
set pg_conndata [dict get $PG_CONNDATA $PG_CONN_DEFAULT]

# Datenbankname von der Kommandozeile
if {$argc > 0} {
    set dbname [lindex $argv 0]
    if {![dict exists $PG_CONNDATA $dbname]} {
        puts "Unbekannte Datenbank: $dbname"
        puts "Datenbanken: [dict keys $PG_CONNDATA]"
        exit 1
    }
    set pg_conndata [dict get $PG_CONNDATA $dbname]
}


### Start
puts "Datenübernahme von: [dict merge $pg_conndata [dict create -password *****]]"

# Passwort für Posgres-db abfragen, falls nicht schon angegeben
if {"[dict get $pg_conndata -password]" == ""} {
    exec stty -echo echonl <@stdin
    puts -nonewline stderr "Passwort fuer [dict get $pg_conndata -user]: "
    gets stdin DB_PASSWORD
    exec stty echo -echonl <@stdin
}
if {"$DB_PASSWORD" == ""} {
    puts "Skriptabbruch wegen fehlendem Passwort"
    exit 1
} else {
    dict set pg_conndata -password $DB_PASSWORD
    unset DB_PASSWORD
}

if {[file exists vlbsettings.sl3db]} {
    puts "vlbsettings.sl3db existiert."
} else {
    puts "vlbsettings.sl3db existiert noch nicht."
}
if {[catch {
    puts "(Re)creating \"vlbsettings.sl3db\" ..."
    exec sqlite3 vlbsettings.sl3db {.read create_vlbsettings_sqlite3.sql}
    puts "\"vlbsettings.sl3db\" created."

    puts "Connecting to \"vlbsettings.sl3db\" ..."
    tdbc::sqlite3::connection create sl3conn "vlbsettings.sl3db"
    puts "Connected to \"vlbsettings.sl3db\""

    puts "Connecting to [dict get $pg_conndata -db] at [dict get $pg_conndata -host] ..."
    tdbc::postgres::connection create pgconn {*}$pg_conndata
    puts "Connected to [dict get $pg_conndata -db]."

    ### Sprachindizes übernehmen und weitergeben
    # Prepared statement für INSERT
    set insert_sprachindex [sl3conn prepare "INSERT INTO sprachindizes (sprachindex_id, \"index\", code, auswahltext) VALUES (:sprachindex_id, :index, :code, :auswahltext)"]
    pgconn foreach sprachindex "SELECT sprachindex_id, index, code, auswahltext FROM sprachindizes" {
        $insert_sprachindex execute $sprachindex
    }

    ### Fragen übernehmen und weitergeben
    #   Blöcke
    # Prepared statement für INSERT
    set insert_frageblock [sl3conn prepare "INSERT INTO fragebloecke (frageblock_id, pos_nr, code, texte) VALUES (:frageblock_id, :pos_nr, :code, :texte)"]
    set frageblock_ids [list]
    pgconn foreach frageblock "SELECT frageblock_id, pos_nr, code, array_to_json(texte) AS texte FROM fragebloecke WHERE code IN ('koerper', 'gewohnheiten') ORDER BY pos_nr" {
        lappend frageblock_ids [dict get $frageblock frageblock_id]
        puts $frageblock
        $insert_frageblock execute $frageblock
    }
    #   Fragen
    # Prepared statement für INSERT
    set insert_frage [sl3conn prepare "INSERT INTO fragen (frage_id, frageblock_id, pos_nr, antworttyp, code, texte) VALUES (:frage_id, :frageblock_id, :pos_nr, :antworttyp, :code, :texte)"]
    set frage_ids [list]
    pgconn foreach frage "SELECT frage_id, frageblock_id, pos_nr, antworttyp, code, array_to_json(texte) AS texte FROM fragen WHERE frageblock_id IN ([join $frageblock_ids ,]) ORDER BY frageblock_id, pos_nr" {
        lappend frage_ids [dict get $frage frage_id]
        puts $frage
        $insert_frage execute $frage
    }
    #   Antwortvorgaben
    # Prepared statement für INSERT
    set insert_antwortvorgabe [sl3conn prepare "INSERT INTO antwortvorgaben (antwortvorgabe_id, frage_id, pos_nr, code, standard, texte) VALUES (:antwortvorgabe_id, :frage_id, :pos_nr, :code, :standard, :texte)"]
    pgconn foreach antwortvorgabe "SELECT antwortvorgabe_id, frage_id, pos_nr, code, CASE WHEN standard THEN 1 ELSE 0 END AS standard, array_to_json (texte) AS texte FROM antwortvorgaben WHERE frage_id IN ([join $frage_ids ,]) ORDER BY frage_id, pos_nr" {
        puts $antwortvorgabe
        $insert_antwortvorgabe execute $antwortvorgabe
    }

    ### Applikationsdaten einfügen
    # Prepared statement für nextids
    set update_nextids [sl3conn prepare "UPDATE nextids SET next_id=next_id+1 WHERE table_name=:table_name"]

    # Die Tabellen zu den Applikationen werden von diesen selbst initialisiert.

    puts "Closing connection to [dict get $pg_conndata -db] ..."
    pgconn close
    puts "Connection to [dict get $pg_conndata -db] closed."
    puts "Closing connection to \"vlbsettings.sl3db\" ..."
    sl3conn close
    puts "Connection to \"vlbsettings.sl3db\" closed."

    puts "\"vlbsettings.sl3db\" created."
} msg]} {
    puts "Error: $msg"
    exit 1
}

