# custom/kernel/recorder.tcl
#
# Recorder zum Aufzeichnen und Abspielen (binärer) Druckdatenstreams
# (Erweiterung des für Arduino control2 definierten Datenformats)
#
# Das Datenformat wird hier syntaktisch definiert.
# Die semantische Interpretation obliegt der jeweiligen App.
#
#{{{ Das Datenfomat einer Aufzeichnung (<recording>)
# (Konstanten sind Hexbytes)
# <recording>:      [<header>]<stream>
# <header>:         <utf8-text>
# <utf8-text>:      Eine oder mehrere Zeilen, (Zeilen)Ende mit '\n'
# <stream>:         [<record>]*
# <record>:         <start><description><data>
# <start>:          ffffffff
# <description>:    <id><dvalue><end>[<id><dvalue><end>]*'T'<timestamp>
# <id>:             <id16>|<id8>
# <id16>:           'A' ... 'S', 'U' ... 'W', 'X'
#                   'A' für Rohwerte der Livedaten
# <id8>:            'a' ... 'w', 'x'
#                   'a' für normierte Werte der Livedaten
#                   Die Bedeutung der Ids ab 'B' bzw. 'b' wird durch die
#                   jeweilige App definiert.
#                   'X' bzw. 'x' ist für Datenblöcke ohne relevanten Inhalt vorgesehen.
# <dvalue>:         <n_rows><n_cols>
# <n_rows>:         8 Bit Unsigned (1 ... 254)
# <n_cols>:         8 Bit Unsigned (1 ... 254)
# <timestamp>:      <d><d><d><d><d><d><d><d>
#                   (ms, bei 0 beginnend, Der 1. Datensatz kann bereits verzögert sein.)
# <d>:              4 Bit (0 ... 9)
# <end>:            ffff
# <data>:           <block>[<block]*
#                   (Die Anzahl der Blöcke ergibt sich aus <description>.)
# <block>:          {<block16>|<block8>}<check><end>
# <block16>:        <value16>[value16>]*
# <block8>:         <value8>[<value8>]*
# <value16>:        15 Bit Big Endian (0 ... 32767 bzw. 0x0000 ... 0x7fff)
# <value8>:         8 Bit Unsigned (0 ... 254 => 0xff ausgeschlossen)
# <check>:          <x><x><x><x>
#                   (XOR über alle Werte; wird beim Lesen von Dateien nicht geprüft)
# <x>:              '0' ... 'f' (Hex-Digit als ASCII)
#}}} Datenfomat
#
# Dateinamen und Verzeichnisse
# /var/local/recordings/[<app>/]<name>.bps ("Binary Pressure Stream")
# /var/local/recordings/[<app>/]<name>.bpsn ("Binary Pressure Stream Notes")
#
#{{{ Datenformat der Notizen (<notes>)
# <notes>:          <note>*
# <note>:           <timestamp> ':' <utf8-text> '\f'
# <timestamp>:      '0'-'9' ['0'-'9']
#                   (max. 8 Stellen)
# <utf8-text>:      utf8-Zeichen außer '\f'
#}}} Datenformat der Notizen
# Anmerkung:
#   Die Notizen werden im Hauptspeicher verwaltet und (ähnlich der Linux command history)
#   nur bei load (bzw. start) initialisiert und bei stop gespeichert.
#
# (Noch) nicht implementiert:
#   proc record: Automatisches Ergänzen fehlender Blöcke
#
# Historie:
# 08.01.2024 Siegmar Müller Begonnen
# 08.02.2024 Siegmar Müller play funktioniert
# 09.02.2024 Siegmar Müller loop und pause funktionieren
# 12.02.2024 Siegmar Müller moveto implementiert
# 16.02.2024 Siegmar Müller Alle Funktionen zum Abspielen fertig
# 06.03.2024 Siegmar Müller Kommandos für Notizen zu einer Aufzeichnung fertig
# 07.03.2024 Siegmar Müller Fertig
#

namespace eval Recorder {

    # Daten zur aktuellen Aufzeichnung
    namespace eval recording {; #{{{
        # Dateiname: /var/local/recordings/[<appname>/]<name>.bps
        variable name ""; # "" <=> keine aktuelle Aufzeichnung geladen
                          # (Aber evtl. $fd_write aktiv)
        variable appname ""
        variable blocks [list]; # Liste der Blockbeschreibungen
                # block: dict mit den folgenden Schlüsseln
                #  id           id wie im 1. Datensatz
                #  n_rows       Anzahl Zeilen
                #  n_cols       Anzahl Spalten
                # Nur beim Lesen:
                #  offs_id      Offset der id in der Beschreibung
                #  offs_values  Offset der Werte
                #  size         Gesamtlänge der Werte in Bytes
        ## Nur für die Aufzeichnung
        variable fd_write ""
        variable ts0 0;     # Beginn der Aufzeichnung (bei start)
        ## Nur zum Abspielen
        # Bei load (einmalig) gesetzt
        variable fp_start 0; # Fileposition des 1. Datensatzes
        variable fp_last 0; # Fileposition des letzten Datensatzes
        variable reclen 0;  # Länge des gesamten Datensatzes
        variable n_records 0; # Anzahl Daten sätze
        variable offs_ts 0; # Offset des Wertes vom Timestamp im Datensatz
        variable duration 0; # Dauer der Aufzeichnung in ms (Timestamp des letzten Datensatzes)
        # Beim Abspielen aktualisiert
        variable loop 0;    # Aufzeichnung in einer Schleife abspielen
        variable pause 0;   # Abspielen pausiert
        variable fd_read ""
        variable record ""; # Der zuletzt gelesene Datensatz
        variable pos_record 0; # Position des zuletzt gelesenen und gesendeten Datensatzes
        variable send_after_id ""; # after id für sendNextRecord
        # Einstellungen
        variable speed 1; # Abspielgeschwindigkeit (Faktor für Zeitlupe/-raffer)
        ## Für Aufzeichnung und Abspielen
        # Einstellungen
        variable POSREFRESH 1000; # Aktualisierung der Position frühestens nach $POSREFRESH ms
        # Steuerung
        variable ts_last 0;         # Timestamp des zuletzt gelesenen/geschriebenen Datensatzes
        variable ts_last_send 0;    # Timestamp des zuletzt gesendeten Datensatzes
        variable ts_last_note -1;   # Timestamp der beim Abspielen zuletzt aufgetauchten Anmerkung
        variable send_pos 1;        # Nächste Positionsmeldung senden
        variable ts_last_pos_change 0;  # Timestamp beim letzten sendPosChange
        # Notizen
        variable notes [dict create];   # Schlüssel ts und note
        variable notes_filename "";     # Dateiname für die Notizen
        #}}}
    }; # namespace eval recording

    variable image_handler_procnames [list]; # Weitergabe von Druckbilddaten
    variable pos_handler_procnames [list]; # Weitergabe der aktuellen Abspielposition
    variable note_handler_procnames [list]; # Weitergabe von Anmerkungen während des Abspielens


    ###{{{ Öffentliche "Bibliotheksprozeduren" und Statusabfragen

    # Timestamp in ASCII umwandeln
    # @param timestamp  Zeitstempel in ms
    # @param hours      0 Stunden auch angeben
    # @return  Zeitstempel im Format hh:mm:ss
    proc ts2asc {timestamp {hours 1}} {; #{{{
        set s [expr "$timestamp / 1000"]
        if {$timestamp % 1000 >= 500} {; # aufrunden
            incr s
        }
        set m [expr "$s / 60"]
        set s [expr "$s % 60"]
        set h [expr "$s / 60"]
        set m [expr "$m % 60"]
        if {!$hours && $h == 0} {
            return [format %02d:%02d $m $s]
        }
        return [format %02d:%02d:%02d $h $m $s]
        #}}}
    }; # proc ts2asc 


    # ASII Timestamp in ms umwandeln
    # @param ascii  Zeitstempel im Format [[h]h:][m]m:ss
    # @return   Zeitstempel in ms oder -1 bei Formatfehler
    proc asc2ts {ascii} {; #{{{
        if {![regexp {^([0-9]?[0-9]:)?[0-9]?[0-9]:[0-9][0-9]$} $ascii]} {; # Formatfehler
            return -1
        }
        set splitted [split $ascii :]
        set timestamp 0
        for {set i 0} {$i < [llength $splitted]} {incr i} {
            set timestamp [expr $timestamp * 60]
            incr timestamp [lindex $splitted $i]
        }
        return [expr $timestamp * 1000]
        #}}}
    }; # proc asc2ts 


    # Ist der Recorder gerade mit Abspeilen oder Aufzeichnen beschäftigt?
    # @return   dict mit appname, recording und what (playing, recording), falls beschäftigt
    #           sonst leeres dict
    proc isBusy {} {; #{{{
        namespace eval recording {
            if {"$fd_read" != "" || "$fd_write" != ""} {
                set what "playing"
                if {"$fd_write" != ""} {
                    set what "recording"
                }
                return [dict create appname $recording::appname recording $recording::name what $what]
            }
            return [dict create]
        }
        #}}}
    }; # proc isBusy 

    ###}}} Öffentliche "Bibliotheksprozeduren"


    # (Weiteren) Handler für gelesenes Druckbild hinzufügen
    # Der Handler muß die folgenden Argumente entgegennehmen: id values n_rows n_cols
    #   Bei $id = "!" enthält $values eine Fehlermeldung und das Abspielen wurde abgebrochen.
    # @param cb_dbld_handler    Vollständiger Name der Callbackprozedur (mit namespace Pfad)
    proc addImageHandler {cb_dbld_handler} {; #{{{
        variable image_handler_procnames

        # Vorsichtshalber prüfen, ob es den schon gibt
        if {[lsearch $image_handler_procnames $cb_dbld_handler] < 0} {
            lappend image_handler_procnames $cb_dbld_handler
            srvLog "[namespace current]::addImageHandler" Debug "$cb_dbld_handler hinzugefügt"
        }
        #}}}
    }; #proc addImageHandler 


    # Handler für gelesenes Druckbild entfernen
    # @param cb_dbld_handler    Der zu entfernende Handler
    # @param force              Keine Warnung wenn der nicht gefunden wurde
    proc removeImageHandler {cb_dbld_handler {force 0}} {; #{{{
        variable image_handler_procnames

        set i [lsearch $image_handler_procnames $cb_dbld_handler]
        if {$i >=0} {
            set image_handler_procnames [lreplace $image_handler_procnames $i $i]
            srvLog [namespace current] Debug "::removeImageHandler $cb_dbld_handler entfernt"
        } elseif {!$force} {
            srvLog [namespace current] Warn "::removeImageHandler $cb_dbld_handler nicht gefunden"
        }
        #}}}
    }; # proc removeImageHandler 


    # (Weiteren) Handler für aktuelle Abspielposition hinzufügen
    # Der Handler muß die Argumente position (0 ... 100) und timestamp [ms] entgegennehmen.
    # @param cb_pos_handler    Name der Handlerprozedur
    proc addPosHandler {cb_pos_handler} {; #{{{
        variable pos_handler_procnames

        # Vorsichtshalber prüfen, ob es den schon gibt
        if {[lsearch $pos_handler_procnames $cb_pos_handler] < 0} {
            lappend pos_handler_procnames $cb_pos_handler
            srvLog "[namespace current]::addPosHandler" Debug "$cb_pos_handler hinzugefügt"
        }
        #}}}
    }; #proc addPosHandler 


    # Handler für aktuelle Abspielposition entfernen
    # @param cb_pos_handler    Der zu entfernende Handler
    # @param force              Keine Warnung wenn der nicht gefunden wurde
    proc removePosHandler {cb_pos_handler {force 0}} {; #{{{
        variable pos_handler_procnames

        set i [lsearch $pos_handler_procnames $cb_pos_handler]
        if {$i >=0} {
            set pos_handler_procnames [lreplace $pos_handler_procnames $i $i]
            srvLog [namespace current]::removePosHandler Debug "$cb_pos_handler entfernt"
        } elseif {!$force} {
            srvLog [namespace current]::removePosHandler Warn "$cb_pos_handler nicht gefunden"
        }
        #}}}
    }; # proc removePosHandler 


    # (Weiteren) Handler für eine beim Abspielen aufgetauchte Anmerkung hinzufügen
    # Der Handler muß die Argumente timestamp [ms] und note entgegennehmen.
    # @param cb_note_handler    Name der Handlerprozedur
    proc addNoteHandler {cb_note_handler} {; #{{{
        variable note_handler_procnames

        # Vorsichtshalber prüfen, ob es den schon gibt
        if {[lsearch $note_handler_procnames $cb_note_handler] < 0} {
            lappend note_handler_procnames $cb_note_handler
            srvLog "[namespace current]::addNoteHandler" Debug "$cb_note_handler hinzugefügt"
        }
        #}}}
    }; #proc addNoteHandler 


    # Handler für eine beim Abspielen aufgetauchte Anmerkung hinzufügen
    # @param cb_note_handler    Der zu entfernende Handler
    # @param force              Keine Warnung wenn der nicht gefunden wurde
    proc removeNoteHandler {cb_note_handler {force 0}} {; #{{{
        variable note_handler_procnames

        set i [lsearch $note_handler_procnames $cb_note_handler]
        if {$i >=0} {
            set note_handler_procnames [lreplace $note_handler_procnames $i $i]
            srvLog [namespace current]::removeNoteHandler Debug "$cb_note_handler entfernt"
        } elseif {!$force} {
            srvLog [namespace current]::removeNoteHandler Warn "$cb_note_handler nicht gefunden"
        }
        #}}}
    }; # proc removeNoteHandler 


    # Positionsänderung an die registrierten Handler senden
    # Die Änderung wird tatsächlich nur dann gesendet,
    # wenn seit dem letzten Senden mindestens $POSREFRESH ms vergangen sind.
    # Beim Aufzeichnen ist die aktuelle Position immer 100%.
    # @param timestamp  Aktueller Zeitstempel in ms
    # @param force      Weitergabe erzwingen
    proc sendPosChange {timestamp {force 0}} {; #{{{
        namespace upvar recording send_pos send_pos duration duration
        namespace upvar recording fd_write fd_write
        namespace upvar recording POSREFRESH POSREFRESH
        namespace upvar recording ts_last_pos_change ts_last_pos_change 
        variable pos_handler_procnames

        if {!($force || $send_pos)} {
            return
        }
        # Beim Aufzeichnen ist die aktuelle Position immer 100%.
        if {"$fd_write" != ""} {
            set pos 100
        } else {
            set pos [expr round ($timestamp * 100 / $duration.0)]
        }
        foreach pos_handler_procname $pos_handler_procnames {
            $pos_handler_procname $pos $timestamp
        }
        set send_pos 0
        set ts_last_pos_change $timestamp
        after $POSREFRESH "set [namespace current]::recording::send_pos 1"
        #}}}
    }; # proc sendPosChange 


    # Ruft sendPosChange mit $ts_last_pos_change + $POSREFRESH auf
    # und wiederholt dies solange $delay_remaining > $POSEFRESH ist.
    # @param delay_remaining    Verbleibende Verzögerung
    proc checkPosChangeDelay {delay_remaining} {; #{{{
        namespace upvar recording POSREFRESH POSREFRESH
        namespace upvar recording ts_last_pos_change ts_last_pos_change 
        namespace upvar recording duration duration

        set ts [expr $ts_last_pos_change + $POSREFRESH]
        if {$ts > $duration} {
            set ts $duration
        }
        sendPosChange $ts
        incr delay_remaining -$POSREFRESH 
        if {$delay_remaining > $POSREFRESH} {
            after $POSREFRESH "[namespace current]::checkPosChangeDelay $delay_remaining"
        }
        #}}}
    }; # proc checkPosChangeDelay 


    # Blöcke des gelesenen Datensatzes senden (an die registrierten Handler verteilen),
    # evtl. vorhandene Anmerkung über Handler weitergeben
    # und nächsten Datensatz lesen
    proc sendNextRecord {} {; #{{{
        namespace upvar recording record record pos_record pos_record blocks blocks
        namespace upvar recording pause pause ts_last ts_last ts_last_send ts_last_send
        namespace upvar recording notes notes ts_last_note ts_last_note 
        variable image_handler_procnames
        variable note_handler_procnames

        set errmsg ""
        set recording::send_after_id ""
        foreach block $blocks {; #{{{ Blöcke durchgehen
            binary scan $record "@[dict get $block offs_id]acc" id n_rows n_cols
            if {[string is ascii $id]} {; # bisjetzt kein Datenmüll
                if {[string is upper $id]} {
                    # 16 Bit
                    if {![string is upper [dict get $block id]]} {
                        set errmsg "Id mismatch at [format 0x%x [expr $pos_record + [dict get $block offs_id]]]"
                        break
                    }
                    set fmt "S"
                } elseif {[string is lower $id]} {
                    # 8 Bit
                    set fmt "c"
                } else {; # Datenmüll
                    set errmsg "Invalid id ('$id')"
                    break
                }
                set length [expr "$n_rows * $n_cols"]
                if {$n_rows * $n_cols != [dict get $block n_rows] * [dict get $block n_cols]} {
                    set errmsg "Invalid block size for '$id' in record at [format 0x%x $pos_record]"
                    break
                }
                binary scan $record "@[dict get $block offs_values]${fmt}${length}" values
                if {$fmt == "c"} {
                    set values2 [list]
                    foreach value $values {
                        if {$value < 0} {
                            lappend values2 [expr $value + 256]
                        } else {
                            lappend values2 $value
                        }
                    }
                    set values $values2
                }
                # Block an die Handler verteilen
                foreach image_handler_procname $image_handler_procnames {
                    $image_handler_procname $id $values $n_rows $n_cols
                }
            } else {; # Datenmüll
                scan $id %c xid; # Bytecode in die Fehlermeldung
                set errmsg "Invalid id ([format 0x%x $id])"
                break
            }
            #}}}
        }; # Blöcke durchgehen

        if {"$errmsg" != ""} {
            srvLog [namespace current]::sendNextRecord Error "$errmsg"
            # Fehler melden
            foreach image_handler_procname $image_handler_procnames {
                $image_handler_procname "!" $errmsg 1 [string length $errmsg]
            }
            stop
            return; # => Abbruch
        }
        # Positionsänderung senden
        sendPosChange $ts_last
        set ts_last_send $ts_last
        # Gibt es zu dem Timestamp eine Anmerkung?
        if {[dict exists $notes $ts_last]} {; # Handler benachrichtigen
            foreach {note_handler_procname} $note_handler_procnames {
                $note_handler_procname $ts_last [dict get $notes $ts_last]
            }
            set ts_last_note $ts_last
        }
        if {$pause} {
            return
        }
        readNextRecord
        #}}}
    }; # proc sendNextRecord 


    # Nächsten Datensatz aus Aufzeichnung lesen und Senden
    #   mit Verzögerung gemäß Timestamp veranlassen
    proc readNextRecord {} {; #{{{
        namespace upvar recording fd_read fd_read reclen reclen offs_ts offs_ts
        namespace upvar recording record record pos_record pos_record ts_last ts_last
        namespace upvar recording ts_last_pos_change ts_last_pos_change 
        namespace upvar recording loop loop speed speed fp_start fp_start
        namespace upvar recording POSREFRESH POSREFRESH

        # Gesamten Datensatz lesen
        set pos_record [tell $fd_read]
        set record [read $fd_read $reclen]
        if {[eof $fd_read]} {
            
            if {$loop} {; # => von vorn
                seek $fd_read $fp_start
                set pos_record [tell $fd_read]
                set record [read $fd_read $reclen]
                # eof unmöglich, weil load bereits mindestens einen Datensatz verlangt.
                set ts_last 0
                sendPosChange 0 1
                srvLog [namespace current]::readNextRecord Info "Loop restart."
            } else {
                sendPosChange $ts_last 1
                stop
                return
            }
        }
        # binary scan (nur Timestamp)
        binary scan $record "@${offs_ts}H8" tsvalue
        scan $tsvalue %d ts
        set delay [expr "$ts - $ts_last"]
        # delay gemäß $speed anpassen
        if {$speed != 1} {
            set delay [expr round($delay / double($speed))]
            if {$speed > 1} {; # Das könnte zu schnell werden.
                if {$delay < 55} {; # Mehr als 18 Bilder/sec
                    # => Bild weglassen, d.h. sofort nächstes lesen
                    after 0 [namespace current]::readNextRecord
                    return
                }
            }
        }
        set ts_last $ts
        set recording::send_after_id [after $delay [namespace current]::sendNextRecord]
        if {$delay > $POSREFRESH} {; # Die Fortschrittsanzeige schläft (vorübergehend) ein.
            # => Mit einer after-Prozedur überbrücken
            set ts_last_pos_change $ts_last
            after $POSREFRESH "[namespace current]::checkPosChangeDelay $delay"
        }
        #}}}
    }; # proc readNextRecord 


    # Aufzeichnung auf nächstem Datensatz zum übergebenen Timestamp positionieren
    # Die Prozedur arbeitet rekursiv.
    # Die Suche endet, wenn nur noch ein Datensatz übrig geblieben ist.
    # @param fd     Filedescriptor von open
    # @param ts     Timestamp zum Positionieren
    # @n_records    Anzahl der noch zu untersuchenden Datensätze ab der aktuellen Dateiposition
    proc seekTS {fd ts n_records} {; #{{{
        namespace upvar recording reclen reclen offs_ts offs_ts 

        srvLog [namespace current]::seekTS Debug "$n_records records left"
        if {$n_records < 2} {
            return
        }
        set fp_lower [tell $fd]
        set n_records_left [expr $n_records / 2]
        # Timestamp im Datensatz in der Mitte der Verbliebenen ansteuern
        seek $fd [expr $n_records_left * $reclen + $offs_ts] current
        srvLog [namespace current]::seekTS Debug "Moved from [format 0x%x $fp_lower] to [format 0x%x [tell $fd]]"
        set tsbytes [read $fd 4]
        # Zurück auf den Datensatzanfang
        seek $fd -[expr $offs_ts + 4] current
        binary scan $tsbytes "H8" tsvalue
        scan $tsvalue %d ts_upper
        if {$ts < $ts_upper} {; # Datensatz liegt in der unteren Hälfte
            srvLog [namespace current]::seekTS Debug "$ts < $ts_upper => lower"
            seek $fd $fp_lower
            seekTS $fd $ts $n_records_left
        } else {; # Datensatz liegt in der oberen Hälfte
            srvLog [namespace current]::seekTS Debug "$ts >= $ts_upper => upper"
            seekTS $fd $ts [expr $n_records_left + ($n_records % 2)]
        }
        #}}}
    }; # proc seekTS 


    ### Recorder Kommandos {{{
    # Die Kommandos geben einen Leerstring oder (im Fehlerfall) eine Fehlermeldung zurück.
    # Ergebnisse werden über einen mitgegebenen Variablennamen zurückgeliefert.

    # Aufzeichnung starten (Neuen Datenstrom anlegen)
    # @param name      Name des Datenstroms
    #                  (Wird für den Dateinamen benutzt, aber nicht in recording::name gespeichert)
    # @param blocks    Liste der Beschreibungen (<block>, ...)
    #       <block>:   dict mit id, n_rows, n_cols
    # @param appname   Name der App oder ""
    # @header          Anwendungsspezifischer Dateiheader der .bps Datei
    # @return          Fehlermeldung oder ""
    proc start {name blocks {appname ""} {header ""}} {; #{{{
        namespace upvar recording fd_write fd_write ts0 ts0
        namespace upvar recording notes notes notes_filename notes_filename 

        srvLog [namespace current]::start Debug "appname: '$appname', name: '$name'"
        # Fehler, falls bereits Aufzeichnung läuft oder abgespielt wird
        set busy [isBusy]
        if {[dict size $busy]} {
            return "Recorder is busy ([dict get $busy what] [dict get $busy recording])"
        }
        set dir "/var/local/recordings"
        if {"$appname" != ""} {
            append dir "/$appname"
        }
        file mkdir $dir; # kein Fehler, wenn es das schon gibt
        if {[catch {set fd_write [open "${dir}/${name}.bps" {CREAT WRONLY BINARY TRUNC}]} err_msg]} { 
            return $err_msg
        }
        puts -nonewline $fd_write $header
        if {[string index $header end] != "\n"} {
            puts -nonewline $fd_write "\n"
        }
        # blocks festhalten
        set recording::blocks $blocks
        # Timestamp des Starts
        set ts0 [clock milliseconds]
        sendPosChange 0
        # Leere *.bpsn -Datei anlegen (nicht geöffnet lassen)
        set notes_filename "${dir}/${name}.bpsn"
        if {[catch {set fd_notes [open $notes_filename {CREAT WRONLY TRUNC}]} err_msg]} { 
            return $err_msg
        }
        close $fd_notes
        set notes [dict create]
        srvLog [namespace current]::start Debug "Recording started: '$name'"
        return ""
        #}}}
    }; # proc start


    # Aufzeichnung zum Abspielen laden und 1. Bild holen
    # Es ist ein Fehler, wenn gerade eine Aufzeichnung läuft oder abgespielt wird.
    # @param name               /var/local/recordings/[<app>/]<name>.bps
    # @param v_recording_info   Variablename zum Speichern der Infos zu der Aufzeichnung
    #                           (dict mit den Schlüsseln header, blocks, duration, n_records)
    # @param appname            Name der App oder ""
    # @return                   Fehlermeldung oder ""
    proc load {name v_recording_info {appname ""}} {; #{{{
        upvar $v_recording_info recording_info
        namespace upvar recording fd_read fd_read fd_write fd_write
        namespace upvar recording notes notes notes_filename notes_filename 

        # Fehler, wenn gerade Aufzeichnung läuft oder abgespielt wird
        set busy [isBusy]
        if {[dict size $busy]} {
            return "Recorder is busy ([dict get $busy what] [dict get $busy recording])"
        }
        set dir "/var/local/recordings"
        if {"$appname" != ""} {
            append dir "/$appname"
        }
        if {![file exists "${dir}/${name}.bps"]} {
            return "Recording '$name' not found in $dir."
        }
        set recording_info [dict create header "" blocks {{id ? n_rows 0 n_cols 0}} duration 0 n_records 0]
        set error_msg ""; # leer => kein Fehler
        if {[catch {; #{{{
            set fd [open "${dir}/${name}.bps" {RDONLY BINARY}]
            # Start des 1. Datenblocks suchen und diesen analysieren
            set header ""
            set ffcount 0
            while {1} {; #{{{ Header vom 1. Datensatz durchgehen
                set c [read $fd 1]
                if {[eof $fd]} {; # Keine Daten
                    set error_msg "Recording is empty."
                    break
                }
                binary scan $c c b
                if {$b == -1} {; # Start vom 1. Datensatz
                    if {[incr ffcount] == 4} {; #{{{ Beginn Daten, 1. Datensatz
                        set fp_start [expr "[tell $fd]" - 4]; # Startposition festhalten
                        srvLog [namespace current]::load Debug "${name}: 1st record found at [format "%x" $fp_start]"
                        # => Beschreibung auswerten
                        set ffcount 0
                        set blocks [list]
                        set datalen 0; # Berechnete Gesamtlänge aller Blöcke
                        while {1} {; # Beschreibung durchgehen
                            set c [read $fd 1]; # id für Descriptor oder Ende
                            if {[eof $fd]} {
                                srvLog [namespace current]::load Debug "${name}: unexpected eof"
                                set error_msg "Invalid file; pos=eof"
                                break
                            }
                            binary scan $c c b
                            if {$b == -1} {; #{{{ Beschreibungsende?
                                # weiteres ff muß folgen
                                set c [read $fd 1]; # id für Descriptor oder Ende
                                binary scan $c c b
                                if {$b != -1} {; # Fehler
                                    srvLog [namespace current]::load Debug "${name}: invalid end of description"
                                    set error_msg "Invalid file; pos=[format "%x" [tell $fd]]"
                                }
                                # Beschreibungsende, wenn hier angekommen
                                srvLog [namespace current]::load Debug "${name}: start of data at 0x[format "%x" [tell $fd]]"
                                # Beschreibung erfolgreich gelesen.
                                break
                                #}}}
                            }; # if {Beschreibungsende?}
                            # Kein Beschreibungsende, wenn hier angekommen
                            set id $c
                            srvLog [namespace current]::load Debug "${name}: id=$id"
                            if {$id == "T"} {; #{{{ Timestamp
                                # Offset zum Lesen der weiteren Timestamps festhalten.
                                set offs_ts [expr [tell $fd] - $fp_start]
                                srvLog [namespace current]::load Debug "${name}: offs_ts: 0x[format "%x" $offs_ts]"
                                # Festhalten für sendPosChange
                                set timestamp [read $fd 4]
                                binary scan $timestamp H8 tsvalue
                                scan $tsvalue %d ts_rec1
                                #}}}
                            } else {; #{{{ Sonstiger Beschreibungswert
                                # Die aktuelle Position ist nach der Id.
                                set offs_id [expr [tell $fd] - $fp_start - 1]
                                set dvalue [read $fd 2]
                                if {[string is ascii $id] && [string is alpha $id]} {; # gültige Id
                                    binary scan $dvalue cc n_rows n_cols
                                    # Zugehörige Datenblocklänge ermitteln
                                    # (Zu den Werten kommen <check> und <end>.)
                                    if {[string is upper $id]} {; # 16 Bit
                                        set size [expr "$n_rows * $n_cols * 2 + 4 + 2"]
                                    } else {; # 8 Bit
                                        set size [expr "$n_rows * $n_cols + 4 + 2"]
                                    }
                                    # Wir brauchen noch offs_values = $datalen + $descrlen 
                                    # Letzteres ist allerdings noch nicht bekannt und
                                    # wird deshalb am Schluß ergänzt.
                                    lappend blocks [dict create id $id n_rows $n_rows n_cols $n_cols offs_id $offs_id size $size offs_values 0]
                                    incr datalen $size
                                } else {; # ungültige Id
                                    srvLog [namespace current]::load Error "${name}: Invalid id ('$id', $b)"
                                    set error_msg "Invalid id ('$id', $b)"
                                    break
                                }
                                #}}}
                            }; # sonstiger Beschreibungswert
                        }; # Beschreibung durchgehen
                        # Beschreibungslänge einschl. <start> und <end>
                        set descrlen [expr "[tell $fd] - $fp_start"]
                        # Gesamtlänge eines Datensatzes
                        set reclen [expr "$descrlen + $datalen"]
                        srvLog [namespace current]::load Debug "reclen=0x[format "%x" $descrlen]+0x[format "%x" $datalen]=[format "%x" $reclen]"
                        #  offs_values = $datalen + $descrlen in allen Blöcken ergänzen
                        set datalen 0
                        dict set recording_info blocks [list]
                        foreach block $blocks {
                            dict set block offs_values [expr $datalen + $descrlen]
                            dict lappend recording_info blocks $block
                            incr datalen [dict get $block size]
                        }
                        break
                        #}}}
                    }; # if {Beginn Daten, 1. Datensatz}
                } elseif {$ffcount > 0} {
                    srvLog [namespace current]::load Debug "${name}: invalid start of description"
                    set error_msg "Invalid file; pos=[tell $fd]"
                    break
                } else {
                    append header $c
                }
                #}}}
            }; # Header vom 1. Datensatz durchgehen

            # Der Header vom 1. Datensatz ist analysiert.
            if {"$error_msg" == ""} {; #{{{ Dabei ist kein Fehler aufgetreten.
                set headerlen [string length $header]
                srvLog [namespace current]::load Debug "headerlen=0x[format "%x" $headerlen]"
                dict set recording_info header $header
                # Analyse des 1. Datenblocks ist abgeschlossen
                srvLog [namespace current]::load Debug "${name}: reclen=$reclen start=$fp_start"
                # Länge der Aufzeichnung ermitteln
                seek $fd -$reclen end; # Auf letzten Datensatz positionieren
                set fp_last [tell $fd]
                set contentlength [expr $fp_last - $fp_start]
                set n_records [expr $contentlength / $reclen + 1]
                set rest [expr $contentlength % $reclen]
                srvLog [namespace current]::load Debug "${name}: $n_records records (Rest $rest)"
                if {$rest > 0} {; # Unsauberes Dateiende
                    error "Suspicious end of file. ($rest additional Bytes)"
                }
                # Anzahl Datensätze festhalten
                dict set recording_info n_records $n_records
                # Timestamp aus letztem Datensatz holen.
                seek $fd [expr $offs_ts - 1] current
                set timestamp [read $fd 5]
                binary scan $timestamp aH8 id tsvalue
                if {$id != "T"} {; # nicht wie erwartet am Timestamp gelanded
                    error "Timestamp of last record not at expected position."
                }
                scan $tsvalue %d ts_last
                srvLog [namespace current]::load Debug "${name}: ts_last=$ts_last ms, [expr $ts_last / 1000] s"
                dict set recording_info duration $ts_last
                # Ersten Datensatz zum Verteilen laden
                seek $fd $fp_start
                set pos_record [tell $fd]
                set record [read $fd $reclen]
                #}}}
            }; # if {kein Fehler in while}
            #}}}
            } msg]} {; # Fehler
                set error_msg $msg
                srvLog [namespace current]::load Error $error_msg
        }; # if {catch ...}
        catch {close $fd}
        if {"$error_msg" == ""} {; # Kein Fehler
            #{{{ => Daten übernehmen
            set recording::name $name
            set recording::appname $appname
            set recording::fp_start $fp_start
            set recording::reclen $reclen
            set recording::offs_ts $offs_ts
            set recording::fp_last $fp_last
            set recording::blocks [dict get $recording_info blocks]
            # Statusvariablen zurücksetzen
            set recording::ts_last $ts_rec1;
            # Steuervariablen zurücksetzen
            set recording::loop 0
            set recording::pause 1; # sendNextRecord soll nur das 1. Bild holen
            set recording::record $record
            set recording::pos_record $pos_record
            set recording::n_records $n_records 
            set recording::duration [dict get $recording_info duration]
            #}}} Daten übernehmen
            sendNextRecord
            set recording::pause 0
            set recording::ts_last_note -1
            set notes_filename "${dir}/${name}.bpsn"
            # notes holen
            set notes [dict create]
            if {[file exists $notes_filename]} {
                if {[catch {
                        set fp_notes [open $notes_filename {RDONLY}]
                        set notes_raw [read $fp_notes]
                        foreach note [split $notes_raw "\f"] {
                            set note_split [split $note ":"]
                            if {[llength $note_split] < 2} {
                                break
                            }
                            set timestamp [lindex $note_split 0]
                            # Im Textteil könnten durch split ':' verloren gegangen sein.
                            dict set notes $timestamp [string range $note [expr [string length $timestamp] + 1] end]
                        }
                        close $fp_notes
                        srvLog [namespace current]::load Info "[dict size $notes] notes from $notes_filename loaded."
                    } msg]} {; # Fehler
                        set error_msg $msg
                        srvLog [namespace current]::load Error $error_msg
                }               
            }
        }
        return $error_msg
        #}}}
    }; # proc load


    # Katalog (Liste) der gespeicherten Aufzeichnungen erstellen
    # /var/local/recordings/[<app>/]<name>.bps ("Binary Pressure Stream")
    # @param v_catalog  Variablenname für das Ergebnis
    # @param appname    Name der App oder ""
    # @param pattern    glob-pattern (ohne .bps)
    # @return   "" oder Fehlermeldung
    proc cat {v_catalog {appname {}} {pattern *}} {; #{{{
        upvar $v_catalog catalog

        set dir "/var/local/recordings"
        if {"$appname" != ""} {
            append dir "/$appname"
        }
        set files [glob -nocomplain $dir/${pattern}.bps]
        srvLog [namespace current]::cat Debug "[llength $files] .bps files found in $dir"
        set catalog [dict create]
        foreach filename $files {
            set name [lindex [file split [file rootname $filename]] end] 
            set fd [open $filename {RDONLY BINARY}]
            set header ""
            while (1) {
                set c [read $fd 1]
                if {[eof $fd]} {
                    break
                }
                binary scan $c c b
                if {$b == -1} {; # Start vom 1. Datensatz
                    break
                }
                append header $c
            }
            close $fd
            dict set catalog $name $header
        }
        return ""
        #}}}
    }; # proc cat


    # Gespeicherte Aufzeichnung löschen
    # /var/local/recordings/[<app>/]<name>.bps ("Binary Pressure Stream")
    # @param appname    Name der App oder ""
    # @param name       Name der Aufzeichnung
    # @return   ""
    proc delete {{appname {}} name} {; #{{{
        set dir "/var/local/recordings"
        if {"$appname" != ""} {
            append dir "/$appname"
        }
        if {[file exists ${dir}/${name}.bpsn]} {
            file delete ${dir}/${name}.bpsn
            srvLog [namespace current]::delete Info "${dir}/${name}.bpsn deleted"
        }
        if {[file exists ${dir}/${name}.bps]} {
            file delete ${dir}/${name}.bps
            srvLog [namespace current]::delete Info "${dir}/${name}.bps deleted"
        }
        return ""
        #}}}
    }; # proc delete 


    # In der geladenen Aufzeichnung an die angegebene Position gehen
    # Wenn das Abspielen noch nicht gestartet wurde,
    # wird es für 1 Bild gestart und danach in pause versetzt.
    # Wenn Zeitangabe nach dem Ende ist, wird stillschweigend auf dieses zurückgesetzt.
    # @param position   Position in % oder im Format [[h]h:][m]m:ss falls !$exact
    # @param exact      position ist ein Timestamp
    # @return   Fehlermeldung oder ""
    proc moveto {position {exact 0}} {; #{{{
        namespace upvar recording name name appname appname
        namespace upvar recording fd_write fd_write fd_read fd_read
        namespace upvar recording fp_start fp_start ts_last ts_last
        namespace upvar recording ts_last_pos_change ts_last_pos_change 
        namespace upvar recording duration duration n_records n_records
        namespace upvar recording pause pause

        # Aufzeichnung geladen? nein => Fehler
        if {"$name" == ""} {
            return "No recording loaded"
        }
        # Wird gerade aufgezeichnet ? ja => Fehler
        if {"$fd_write" != ""} {
            return "Recorder busy (recording)";
        }
        set dir "/var/local/recordings"
        if {"$appname" != ""} {
            append dir "/$appname"
        }
        set fd [open "${dir}/${name}.bps" {RDONLY BINARY}]
        seek $fd $fp_start
        if {$exact} {
            if {$position < 0 || $position > $duration} {
                return "position (timestamp) out of range."
            }
            set ts_move $position
        } else {
            # Gültige Position ?
            if {[string is integer $position]} {
                if {$position < 0 || $position > 100} {
                    return "Invalid position ($position)"
                }
                set ts_move [expr $duration * $position / 100]
            } else {
                if {[set ts_move [asc2ts $position]] < 0} {
                    return "Invalid position format ($position)"
                }
                if {$ts_move > $duration} {; # Nach dem Ende
                    # => stillschweigend auf das Ende setzen
                    set ts_move $duration
                }
            }
        }
        srvLog [namespace current]::moveto Debug "Timestamp to move to: $ts_move"
        seekTS $fd $ts_move $n_records
        set fp_found [tell $fd]
        srvLog [namespace current]::moveto Debug "Record found at: [format 0x%x $fp_found]"
        if {"$fd_read" == ""} {
            set pause 1
            set fd_read $fd
            srvLog [namespace current]::moveto Debug "fd -> fd_read"
        } else {
            close $fd
            seek $fd_read $fp_found
            srvLog [namespace current]::moveto Debug "seek fd_read [format 0x%x $fp_found]"
        }
        set ts_last $ts_move; # Ohne Verzögerung weitermachen
        set ts_last_pos_change $ts_move
        # Bei pause Datensatz anzeigen
        if {$pause} {
            readNextRecord
        }
        # Position an pos_handler_procnames verteilen
        sendPosChange $ts_last
        return ""
        #}}}
    }; # proc moveto 


    # Die aktuelle Position in der geladenen Aufzeichnung um den angegebenen Betrag verschieben
    # @param offset Verschiebung in s
    # @return   Fehlermeldung oder ""
    proc move {offset} {; #{{{
        namespace upvar recording ts_last ts_last duration duration
        
        # Falls irgendetwas gegen die Neupositionierung spricht, wird das von moveto bemerkt.
        set ts_move [expr $ts_last + $offset * 1000]
        if {$ts_move < 0} {
            set ts_move 0
        } elseif {$ts_move > $duration} {
            set ts_move $duration
        }
        return [moveto $ts_move 1]
        #}}}
    }; # proc move 


    # Abspielgeschwindigkeit einstellen
    # @param speed  Faktor für die Abspielgeschwindigkeit
    # @return   Fehlermeldung oder ""
    proc setSpeed {speed} {; #{{{
        if {![string is double $speed] || $speed <= 0} {
            return "Invalid speed factor: $speed"
        }
        set recording::speed $speed
        return ""
        #}}}
    }; # proc setSpeed 


    # Abspielen der geladenen Aufzeichnung beginnen bzw. fortsetzen
    # Es ist ein Fehler, wenn gerade aufgezeichnet oder abgespielt wird.
    # Wenn das Abspielen pausiert, wird es wieder aufgenommen.
    # Die Wiederaufnahme erfolgt nur dann in einer Schleife, wenn loop 1 ist,
    # unabhängig davon, wie loop beim vorhergehenden Start gesetzt war.
    # @param loop   1 Abspielen in einer Schleife
    # @return "" oder Fehlermeldung
    proc play {{loop 0}} {; #{{{
        namespace upvar recording name name appname appname
        namespace upvar recording fd_write fd_write fd_read fd_read
        namespace upvar recording fp_start fp_start ts_last ts_last
        namespace upvar recording pause pause blocks blocks
        variable image_handler_procnames 

        # Aufzeichnung geladen? nein => Fehler
        if {"$name" == ""} {
            return "No recording loaded"
        }
        # Wird gerade aufgezeichnet oder schon abgespielt? ja => Fehler
        if {"$fd_write" != ""} {
            return "Recorder busy (recording)";
        }
        if {"$fd_read" != ""} {
            if {$pause} {
                set pause 0
                set recording::loop $loop
                readNextRecord
                return ""
            }
            return "Recorder busy (playing)";
        }
        # Aufzeichnung geladen und Abspielen noch nicht gestartet, wenn hier angekommen
        # => Datei öffnen und Abspielen auf Anfang setzen
        set dir "/var/local/recordings"
        if {"$appname" != ""} {
            append dir "/$appname"
        }
        set fd_read [open "${dir}/${name}.bps" {RDONLY BINARY}]
        #   Letzter Timestamp = 0
        set ts_last 0
        #   Auf 1. Datensatz positionieren
        seek $fd_read $fp_start
        srvLog [namespace current]::play Info "File position set to 0x[format %x $fp_start]."
        set recording::loop $loop
        sendPosChange 0 1; # $ts_last
        # Wenn ts des 1. Records groß ist, ist es besser ein Leerbild einzuschieben,
        # ansonsten stört es nicht.
        foreach block $blocks {; # Blöcke durchgehen
            # Leeren Block an die Handler verteilen
            foreach image_handler_procname $image_handler_procnames {
                set id [dict get $block id]
                set n_rows [dict get $block n_rows]
                set n_cols [dict get $block n_cols]
                set values [lrepeat [expr $n_rows * $n_cols] 0]
                $image_handler_procname $id $values $n_rows $n_cols
            }
        }
        readNextRecord
        return ""
        #}}}
    }; # proc play


    # Abspielen unterbrechen
    # Sendet auch den Timestamp des aktuellen Datensatzes
    # Es ist ein Fehler, wenn gerade nichts abgespielt wird,
    # nicht aber, wenn das Abspielen bereits pausiert.
    # @return "" oder Fehlermeldung
    proc pause {} {; #{{{
        namespace upvar recording fd_read fd_read ts_last ts_last

        if {"$fd_read" == ""} {
            return "Not playing"
        }
        set recording::pause 1
        # Exakte Position schicken
        # (Evtl. soll die für späteres Zurückspringen gespeichert werden.)
        sendPosChange $ts_last 1
        return ""
        #}}}
    }; # proc pause 


    # Aufzeichnung bzw. Abspielen beenden (Datenstrom schließen)
    # @return ""
    proc stop {} {; #{{{
        namespace upvar recording fd_write fd_write fd_read fd_read
        namespace upvar recording pause pause send_after_id send_after_id 
        namespace upvar recording notes notes notes_filename notes_filename

        set stopped 0; # Es wurde tatsächlich etwas gestoppt.
        if {"$fd_read" != ""} {; # Es wird gerade abgespielt.
            # => Nächstes sendNextRecord annullieren
            if {"$send_after_id" != ""} {
                after cancel $send_after_id
                set send_after_id ""
            }
        }
        foreach fd [list fd_write fd_read] {
            if {[set $fd] == ""} {
                continue
            }
            if {[catch {close [set $fd]} msg]} {
                srvLog [namespace current]::stop Warn "Error '$msg' while closing $fd."
            }
            set $fd ""
            srvLog [namespace current]::stop Info "Stopped ($fd)."
            set stopped 1
        }
        set pause 0
        if {$stopped} {
            # Notizen speichern
            if {[catch {
                    set fd_notes [open $notes_filename {CREAT WRONLY TRUNC}]
                    set timestamps [lsort -integer [dict keys $notes]]
                    foreach timestamp $timestamps {
                        puts -nonewline $fd_notes "$timestamp:[dict get $notes $timestamp]\f"
                    }
                    close $fd_notes
                    srvLog [namespace current]::stop Debug "[llength $timestamps] notes stored."
                } err_msg]} {
                    srvLog [namespace current]::stop Error $err_msg
                    return $err_msg
            }
        }
        return ""
        #}}}
    }; # proc stop


    # Recorder deaktivieren
    # @param send_unload    leere Druckbilder senden (1 je Block) 
    #                       und Position auf 0 zurücksetzen
    # @return ""
    proc unload {{send_unload 0}} {; #{{{
        namespace upvar recording fd_read fd_read blocks blocks
        variable image_handler_procnames 

        set was_playing [expr {"$fd_read" != ""}]
        stop
        set recording::name ""
        set recording::appname ""
        set recording::speed 1
        if {$was_playing && $send_unload} {
            foreach block $blocks {; # Blöcke durchgehen
                # Leeren Block an die Handler verteilen
                foreach image_handler_procname $image_handler_procnames {
                    set id [dict get $block id]
                    set n_rows [dict get $block n_rows]
                    set n_cols [dict get $block n_cols]
                    set values [lrepeat [expr $n_rows * $n_cols] 0]
                    $image_handler_procname $id $values $n_rows $n_cols
                }
            }
            sendPosChange 0 1
        }
        srvLog [namespace current]::unload Debug "unloaded"
        return ""
        #}}}
    }; # proc unload 

    ###}}} Recorder Kommandos
    
    # Datensatz aufzeichnen
    # Reihenfolge der Datenblöcke wie bei start
    # (Noch nicht implementiert:
    #       Bei fehlenden Datenblöcken wird id durch X bzw x ersetzt und die Werte sind alle 0.)
    # @param ldata   Liste der Liste der Druckwerte (Anzahl gemäß Definition s. start)
    # @return   Fehlermeldung oder ""
    proc record {ldata} {; #{{{
        namespace upvar recording fd_write fd_write ts0 ts0 ts_last ts_last blocks blocks

        if {[llength $ldata] != [llength $blocks]} {
            set msg "Invalid data lenght ([llength $value]) in block '[dict get $block id]'"
            srvLog [namespace current]::record Error $msg
            return $msg
        }
        if {[catch {
            set ts_now [clock milliseconds]

            # Start 0xffffffff
            puts -nonewline $fd_write [binary format "I" -1]

            # Beschreibunsblock
            foreach block $blocks {
                # Blockstart Mattenkennung, Anzahl Zeilen und Spalten
                set id [dict get $block id]
                set n_rows [dict get $block n_rows]
                set n_cols [dict get $block n_cols]
                puts -nonewline $fd_write [binary format "acc" $id $n_rows $n_cols]
            }
            # Timestamp (8 Dezimaldigits in 4 Bytes)
            set ts_last [expr $ts_now - $ts0]
            puts -nonewline $fd_write [binary format "aH8" "T" [format "%08d" $ts_last]]
            # Ende des Beschreibungsblocks 0xffff
            puts -nonewline $fd_write [binary format "S" -1]
    
            # Datenblock
            for {set i 0} {$i < [llength $blocks]} {incr i} {
                set block [lindex $blocks $i]
                set id [dict get $block id]
                set n_rows [dict get $block n_rows]
                set n_cols [dict get $block n_cols]
                set values [lindex $ldata $i]

                # Druckwerte
                if {[llength $values] != [expr {$n_rows * $n_cols}]} {
                    set msg "Invalid data lenght ([llength $value]) in block '[dict get $block id]'"
                    srvLog [namespace current]::record Error "$msg"
                    return $msg
                }
                if {[string is upper $id]} {; # 16 Bit big endian
                    puts -nonewline $fd_write [binary format "S[expr $n_rows * $n_cols]" $values]
                } else {; # 8 Bit
                    puts -nonewline $fd_write [binary format "c[expr $n_rows * $n_cols]" $values]
                }
                # Check
                set check 0
                foreach value $values {
                    set check [expr $check ^ $value]
                }
                puts -nonewline $fd_write [format "%04x" $check]
                # Datenende 0xffff
                puts -nonewline $fd_write [binary format "S" -1]
            }

            flush $fd_write
            sendPosChange $ts_last
        } msg]} {
            srvLog [namespace current]::record Error "$msg"
            return $msg
        }
        return ""
        #}}}
    }; # proc record 


    # Die Notizen
    # Notizen beziehen sich auf die aktuell geladene oder die laufende Aufzeichnung.


    # Notiz festhalten
    # (Nur bei geladener oder laufender Aufzeichnung)
    # @param note       Die Notiz. Überschreibt eine evtl. bereits vorhandene
    # @return   Fehlermeldung oder ""
    proc writeNote {note} {; #{{{
        namespace upvar recording name name notes notes
        namespace upvar recording ts_last ts_last ts_last_send ts_last_send
        namespace upvar recording fd_write fd_write
        namespace upvar recording ts_last_note ts_last_note 
        variable note_handler_procnames

        if {"$fd_write" != ""} {; # Aufzeichnung läuft
            set timestamp $ts_last
        } elseif {"$name" != ""} {
            set timestamp $ts_last_send
        } else {
            return "No recording loaded"
        }
        dict set notes $timestamp $note
        srvLog [namespace current]::writeNote Debug "Note written: $timestamp $note"
        # Anmerkung verteilen
        foreach {note_handler_procname} $note_handler_procnames {
            $note_handler_procname $timestamp $note
        }
        set ts_last_note $ts_last
        return ""
        #}}}
    }; # proc writeNote 


    # Liste von max. 101 Positionen (Markierungen) an denen sich Notizen befinden erstellen
    # (Nur bei geladener Aufzeichnung)
    # Zu jeder Position gibt es eine Liste von Timestamps unter denen sich die Notes befinden.
    # @param name_markers   Variablenname unter dem die Markierungen abgelegt werden:
    #                       dict mit den Positionen in % als Schlüssel
    #                           Werte sind dicts mit den Schlüsseln
    #                               time (Position im Format hh:mm:ss) und
    #                               timestamps (Liste der Timestamps der Notizen)
    # @return   Fehlermeldung oder ""
    proc getNoteMarks {name_markers} {; #{{{
        upvar $name_markers markers
        namespace upvar recording name name notes notes
        namespace upvar recording duration duration

        if {"$name" == ""} {
            return "No recording loaded"
        }
        set markers [dict create]
        set last_position -1
        foreach timestamp [dict keys $notes] {
            set position [expr $timestamp * 100 / $duration]
            if {$position != $last_position} {
                dict set markers $position ascts [ts2asc $timestamp]
                set timestamps [list]
                set last_position $position
            }
            lappend timestamps $timestamp
            dict set markers $position timestamps $timestamps
        }
        srvLog [namespace current]::getNoteMarks Debug "$markers"
        return ""
        #}}}
    }; # proc getNoteMarks 


    # Sortierte Liste der Timestamps mit Anmerkungen holen
    # (Nur bei geladener Aufzeichnung)
    # @param name_ts    Variablenname unter dem die Timestamps abgelegt werden
    # @return   Fehlermeldung oder ""
    proc getNotesTimestamps {name_ts} {; #{{{
        upvar $name_ts timestamps
        namespace upvar recording name name notes notes

        if {"$name" == ""} {
            return "No recording loaded"
        }
        set timestamps [lsort -integer [dict keys $notes]]
        return ""
        #}}}
    }; # proc getNotesTimestamps 


    # Notiz zu angegebenem Timestamp holen
    # (Nur bei geladener Aufzeichnung)
    # @param timestamp  Timestamp der fraglichen Notiz
    # @param name_note  Variablenname unter dem die Markierungen abgelegt werden
    # @return   Fehlermeldung oder ""
    proc getNote {timestamp name_note} {; #{{{
        upvar $name_note note
        namespace upvar recording name name notes notes

        if {"$name" == ""} {
            return "No recording loaded"
        }
        if {![dict exists $notes $timestamp]} {
            return "No note at $timestamp"
        }
        set note [dict get $notes $timestamp]
        return ""
        #}}}
    }; # proc getNote 

    
    # Notiz löschen
    # (Nur bei geladener Aufzeichnung)
    # @param timestamp  timestamp der zu löschenden Anmerkung
    #                   -1 (nur beim Abspielen) zuletzt aufgetauchte Anmerkung 
    # @return   Fehlermeldung oder ""
    proc delNote {{timestamp -1}} {; #{{{
        namespace upvar recording name name notes notes
        namespace upvar recording ts_last_note ts_last_note
        namespace upvar recording fd_write fd_write

        srvLog [namespace current]::dropNote Debug "timestamp: $timestamp; ts_last_note: $ts_last_note"
        if {"$fd_write" == "" && "$name" == ""} {
            return "No recording loaded"
        }
        if {$timestamp == -1}  {
            if {"$fd_write" != ""} {; # Aufzeichnung läuft
                return "Timestamp must be explicitely given."
            }
            if {$ts_last_note == -1} {
                return "No previousely encountered Note"
            }
            set timestamp $ts_last_note
        }
        if {![dict exists $notes $timestamp]} {
            return "No note found at $timestamp"
        }
        dict unset notes $timestamp
        srvLog [namespace current]::dropNote Info "Note dropped at $timestamp"
        return ""
        #}}}
    }; # proc delNote 


    # Initialisierung nach dem Starten des Moduls
    # (Dummy, um Warnung zu vermeiden.)
    proc init {} {; #{{{
        #}}}
    }; # proc init 

}; # namespace eval Recorder 

set mod_loaded Recorder

