#!/bin/bash
#
# create_vmkstationd_tar_gz.sh
#
# Erzeugt ein Installationspaket für den vmkstationd3
# => create_vmkstationd_vlbupdate_tar_gz.sh erzeugt daraus ein Setup/Update Paket für vlbupdate.
#
# Dieses Skript befindet sich im Unterverzeichnis /devel
# und muß sich während der Ausführung ebenfalls dort befinden.
# Arbeitsverzeichnis ist das "vmkstiond3" enthaltende Verzeichnis.
# Der Aufruf erfolgt ohne Kommandozeilenparameter.
# Der Start erfolgt nach einer Rückfrage. Bei Abbruch wird mit Status 1 beendet.
# Die Ausführung muß durch root erfolgen, damit die Dateien den Besitzer und die Gruppe "root" erhalten können.
# Dazu wird alles temporär in ein Verzeichnis vmkstationd_V<VERSION> kopiert.
# Das Verzeichnis devel/ sowie Backups und Versionierungsdateien werden aus dem Archiv ausgeschlossen.
# Der Verzeichnisname ist gleichzeitig der Basisname des Tarfiles.
# Versionen mit Vorläufigkeitsanhang erhalten das aktuelle Datum.
# Ältere Installationspakete werden nach Rückfrage gelöscht.
#
# Velometrik GmbH
#
# Author: Siegmar Müller
# Historie:
#   28.10.2019  erstellt für vmkstationd
#   09.02.2023  umgebaut für vmkstationd3
#   25.10.2023  Kopfkommentar überarbeitet
#


# Nicht mehr benötigte temporäre Dateien/Verzeichnisse am Ende löschen
CLEANUP=true

# Ausführung als root, damit die Dateien Besitzer und Gruppe "root" bekommen können.
if [ $UID -ne 0 ]; then
    echo "Das Skript muß von root ausgeführt werden."
    su -c $0
    exit $?
fi

echo "Installationspaket ../vmkstationd3/setup/vmkstationd_V<version>.tar.gz erstellen ..."

# Beendet das Skript mit dem angegeben Status
# und stellt zuvor nötigenfalls vmkstationd.tcl wieder her.
exitScript () {
    if [ -f vmkstationd3/vmkstationd.tcl~ ]; then
        cp -fv vmkstationd3/vmkstationd.tcl~ vmkstationd3/vmkstationd.tcl
    fi
    [ $# -eq 0 ] && exit 1
     rm -fv vmkstationd3/vmkstationd.tcl~
    exit $1
}


SCRIPT_DIR=$( cd -- "$( dirname $0 )" > /dev/null && pwd)
if [ "${SCRIPT_DIR:0-19}" != "/vmkstationd3/devel" ]; then
    echo "$( basename $0 ) muß sich im Verzeichnis ../vmkstationd3/devel befinden."
    exit 1
fi
cd $SCRIPT_DIR
cd .. # vmkstationd3
cd ..
echo "Skriptausführung in `pwd`."
# Wir sind nun im Ausführungsverzeichnis.

# Die Versionsnummer wird dem Hauptskript (ohne Datumsanhang) entnommen.
VERSION=`grep '^ *set *VERSION' vmkstationd3/vmkstationd.tcl | sed -E 's/^.*([0-9]+\.[0-9]+\.[0-9]+_?[XAU]?).*$/\1/'`
[ ${PIPESTATUS[0]} -ne 0 ] && exit 1
echo "Gefundene Version: $VERSION"
# Gültige Version ?
echo $VERSION | grep '^[0-9][0-9]*\.[0-9][0-9]*\.[0-9][0-9]*[^ ]*$' > /dev/null
if [ $? -ne 0 ]; then
    echo "Die Version kann nicht ermittelt werden."
    echo "(Die Syntax scheint nicht zu stimmen.)"
    exit 1
fi
if `echo $VERSION | grep -q '_[XAU]$'`; then # Vorläufige Version
    # => aktuelles Datum einbauen.
    VERSION=${VERSION}`date +%Y%m%d`
    # Diese Version in das Skript hineinoperieren
    cp -fv vmkstationd3/vmkstationd.tcl vmkstationd3/vmkstationd.tcl~
    sed -E '/^ *set *VERSION/s/VERSION .*$/VERSION '"$VERSION"'; # modified by create_vmkstationd_tar_gz.sh/' vmkstationd3/vmkstationd.tcl~ > vmkstationd3/vmkstationd.tcl
    echo "Aktuelles Datum in unfertige Version eingebaut: $VERSION"
else
    echo "Fertige Version: $VERSION"
fi

# Alles muß in ein Verzeichnis mit eingeschlossener Versionsnummer gepackt werden.
TMPDIR=vmkstationd_V${VERSION} # nicht vmkstationd3_...
echo "Temp. Verzeichnis für vmkstationd: $TMPDIR"

echo -n "Create ${TMPDIR}.tar from vmkstationd3 (J|n): "
read ANSWER
[ "${ANSWER}" == "" ] && ANSWER="J"
[ "${ANSWER}" != "J" ] && { echo "abgebrochen"; exitScript 1; }

if [ -d $TMPDIR ]; then
    read -p "Altes Verzeichnis ./$TMPDIR löschen?[Jn]: " A
    if [ "$A" == "n" ]; then
        echo "abgebrochen"
        exitScript 1
    fi
    rm -rf ./$TMPDIR
fi

# tarfile erstellen
cp vmkstationd3 -r $TMPDIR
chown -R root $TMPDIR
chgrp -R root $TMPDIR
echo
tar --exclude=${TMPDIR}/devel --exclude-backups --exclude-vcs -cvf ${TMPDIR}.tar ${TMPDIR}
echo
gzip -f ${TMPDIR}.tar
chown root ${TMPDIR}.tar.gz
chgrp root ${TMPDIR}.tar.gz
echo "${TMPDIR}.tar.gz erzeugt"

# tarfile nach vmkstationd3/setup kopieren
# Vorhandene tars vorher löschen
TARS=`ls vmkstationd3/setup/vmkstationd*.tar.gz 2>/dev/null`
if [ ${#TARS} -gt 0 ]; then
    echo -n "Alte tarfile(s) in vmkstationd3/setup/ löschen? (J|n): "
    read ANSWER
    [ "${ANSWER}" == "" ] && ANSWER="J"
    [ "${ANSWER}" != "J" ] && { echo "abgebrochen"; exitScript 1; }
    for TAR in $TARS; do
        rm -v $TAR
    done
fi
mv -v ${TMPDIR}.tar.gz vmkstationd3/setup
chmod a+w vmkstationd3/setup/${TMPDIR}.tar.gz

# vmkstationd.tcl mit ursprünglicher Version wiederherstellen
if [ -f vmkstationd3/vmkstationd.tcl~ ]; then
     cp -fv vmkstationd3/vmkstationd.tcl~ vmkstationd3/vmkstationd.tcl
    [ $? -ne 0 ] && exit 1
     rm -fv vmkstationd3/vmkstationd.tcl~
fi
if `$CLEANUP`; then
    echo "Lösche temporäres Verzeichnis $TMPDIR"
    rm -rf ./$TMPDIR
fi
echo "Installationspaket vmkstationd3/setup/${TMPDIR}.tar.gz erzeugt"

