#!/bin/bash
#
# .../setup/setup_clients.sh
#
# Velometrik GmbH
#
# Installationsskript für lokale Clients des vmkstationd auf der Velobox
# Voraussetzungen:
#   - vmkstationd ist installiert
#   - Ein Display ist vorhanden und X11 ist installiert
# Das Logfile ist /var/local/log/vmksdsetup.log.
#
# Historie:
#   25.04.2023 Siegmar Müller   Begonnen
#


# Hilfe ausgeben
help() {
    echo
    echo "vmkstationd Setup lokaler Clients:"
    echo
    echo "Aufruf: $0 [<optionen>]"
    echo "Optionen:"
    echo "-?    Hilfe ausgeben und beenden"
    if [ $UID -ne 0 ]; then
        echo "(Die Ausführung erfordert root-Rechte.)"
    fi
    echo
}

# Start
echo
echo "$0 $@"
echo

# Optionen aus der Kommandozeile holen.
while getopts ? OPTION; do
    case $OPTION in
        ?) help; exit 0;;
    esac
done

# Ausführung als root
if [ $UID -ne 0 ]; then
    echo "Das Skript benötigt 'root' Rechte."
    su -c "$0 $*"
    exit $?
fi

SETUP="$(cd $(dirname $0) > /dev/null 2>&1; pwd -P)/"
VMKSTATIOND=`echo $SETUP | sed 's/setup\///'`
# In dem Verzeichnis muß vmkstationd.tcl sein:
if [ ! -x ${VMKSTATIOND}vmkstationd.tcl ]; then
    echo "$0 befindet sich nicht in der Umgebung von vmkstationd3."
    exit 1
fi

### Vorbereitung für Logausgaben und Fehlertest mit eventuellem Abbruch

LOGFILE=/var/local/log/vmksdsetup.log

if [ -f $LOGFILE ]; then
    echo -e "\n-------------------------------------------------------------------------------" >> $LOGFILE
else
    # Das Logverzeichnis existiert noch nicht in einem neu aufgesetzten System.
    mkdir -p `dirname $LOGFILE`
fi
echo "setup.sh `date '+%d.%m.%Y %H:%M:%S'`" | tee -a $LOGFILE
SETUP_DIR=`echo \`realpath $0\` | sed 's?/setup.sh??'`
echo "SETUP_DIR = $SETUP_DIR" | tee -a $LOGFILE

# Ausgabe von Logmeldungen in der Form:
#   echo "Meldung" | tee -a $LOGFILE
log() {
    echo "$*" | tee -a $LOGFILE
}


# Setup bei Fehler abbrechen
#   cancelSetup [<exit_status>]
# Ohne Angabe von <exit_status> wird mit Status 0 abgebrochen.
# Bei <exit_status> 0 geht es weiter.
cancelSetup() {
    if [ $# -eq 0 ]; then
        log "setup.sh abgebrochen."
        exit 0
    fi
    if [ $1 -ne 0 ]; then
    	log "setup.sh mit Status $1 abgebrochen"
    	exit $1
    fi
} # cancelSetup()


# Umleitung von Kommandoausgaben:
#   <command> 2>&1 | tee -a $LOGFILE
# Umleitung von Kommandoausgabe und Statusauswertung:
#   <command> 2>&1 | tee -a $LOGFILE
#   [ ${PIPESTATUS[0]} -ne 0 ] &&  cancelSetup 1


# Ist xinit installiert ?
STATUS=`dpkg-query --showformat '${db:Status-Status}' -W xinit 2>&1`
RC=$?
if [ $RC -ne 0 -o "$STATUS" != "installed" ]; then
    log "X11 (xinit) muß erst installiert werden."
    cancelSetup 1
fi

# Debian-Pakete, die die Clients benötigen.
DEBIAN_PAKETE="tk8.6 libtk-img"
for DEB_PAKET in $DEBIAN_PAKETE; do
    STATUS=`dpkg-query --showformat '${db:Status-Status}' -W $DEB_PAKET 2>&1`
    RC=$?
    if [ $RC -eq 2 ]; then
        log "FEHLER: $STATUS"
        cancelSetup 2
    fi
    if [ $RC -eq 1 ]; then
        STATUS="not-installed"
    fi
    if [ "$STATUS" == "installed" ]; then
        log "$DEB_PAKET ist bereits installiert."
    elif [ "$STATUS" == "not-installed" ]; then
        apt-get -y install $DEB_PAKET 2>&1 | tee -a $LOGFILE
    	[ ${PIPESTATUS[0]} -ne 0 ] &&  cancelSetup 1
    else
        log "FEHLER: Unbekannter Status ($STATUS) für Paket $DEB_PAKET."
        cancelSetup 1
    fi
done
if [ -L /usr/bin/wish ]; then
    log "Link /usr/bin/wish existiert bereits."
else
    ln -v -s /usr/bin/wish8.6 /usr/bin/wish 2>&1 | tee -a $LOGFILE
    [ ${PIPESTATUS[0]} -ne 0 ] &&  cancelSetup 1
fi


# Läuft der vmkstationd.sh ?
service vmkstationd.sh status 2>&1 | tee -a $LOGFILE
RC=${PIPESTATUS[0]}
if [ $RC -eq 0 ]; then # Service läuft
	log "vmkstationd läuft."
elif [ $RC -eq 3 ]; then # Service existiert, läuft aber nicht
    log "vmkstationd ist installiert, läuft aber nicht."
    service vmkstationd.sh status 2>&1 | tee -a $LOGFILE
    [ ${PIPESTATUS[0]} -ne 0 ] &&  cancelSetup 1
elif [ $RC -eq 4 ]; then # Service existiert noch nicht
    log "vmkstationd muß erst installiert werden."
    cancelSetup 1
fi


service vlbdisplay.sh status 2>&1 | tee -a $LOGFILE
RC=${PIPESTATUS[0]}
if [ $RC -eq 0 ]; then # Service läuft
	log "vlbdisplay läuft."
elif [ $RC -eq 3 ]; then # Service existiert, läuft aber nicht
    log "vlbdisplay ist installiert, läuft aber nicht."
elif [ $RC -eq 4 ]; then # Service existiert noch nicht
    # Startskript für Velobox Display Daemon kopieren und scharf schalten
    cp -v $SETUP/vlbdisplay.sh /etc/init.d/ 2>&1  | tee -a $LOGFILE
    [ ${PIPESTATUS[0]} -ne 0 ] && cancelSetup 1
    update-rc.d vlbdisplay.sh defaults 2>&1  | tee -a $LOGFILE
    [ ${PIPESTATUS[0]} -ne 0 ] && cancelSetup 1
    log "Service vlbdisplay.sh wird gestartet ..."
    service vlbdisplay.sh start 2>&1  | tee -a $LOGFILE
    [ ${PIPESTATUS[0]} -ne 0 ] && cancelSetup 1
fi

log "Lokale Clients erfolgreich installiert."

