# Anwendung: "Satteldruckanalyse"
# Datei: custom/apps/satteldruckanalyse.tcl
#
# Historie:
# 14.06.2023 Siegmar Müller Version 1.1. mit normierten Druckwerten
#

package require Thread
package require json

#TODO Wird bei jedem "switch -regexp ..." wo nötig der reguläre Ausdruck eindeutig formuliert ?

namespace eval satteldruckanalyse {
    variable VERSION "1.1.0"
    # Die benötigten Fragen aus vmkprodukte werden bei init in vlbsettings eingetragen.
    #TODO 14.1.2023: An Sprachen haben wir aktuell nur deutsch und englisch wie aktuell in vmkprodukte.
    variable APPQUESTIONS [dict create \
                    sitbones { \
                            pos_nr 1 \
                            codes {sitzknochenabstand 1 asymmetrie 2} \
                            texte {Sitzknochen Sitbone} \
                            } \
                    anamnesis { \
                            pos_nr 2 \
                            codes {einsatzbereich 1 fahrleistung 2 sitzposition 3} \
                            texte {Anamnese Anamnesis} \
                            } \
                    ]
    # Die in vlbsettings:applications.clientsettings zu speichernden Einstellungen
    # Diese werden von den diesbezüglichen Kommandos modifiziert.
    # Hier sind die defaults (die ohnehin eingestellt sind, d.h. initial keine Aktion erfordern).
    # variable "appsettings" {{{
    variable appsettings [dict create \
        create_jpeg  {"value" "on" "default" "on"} \
        create_norm  {"value" "off" "default" "off"} \
        maskname {"value" "nomask" "default" "nomask"} \
        compare {"value" "on" "default" "on"} \
        contrast {"value" 0 "default" 0} \
        bgcolor {"value" "#FFFFFF" "default" "#FFFFFF"} \
        grid { \
            0 {"value" "no" "default" "no"} \
            1 {"value" "no" "default" "no"} \
            2 {"value" "no" "default" "no"} \
            3 {"value" "no" "default" "no"} \
            } \
        resolution { \
            0 {"value" 20 "default" 20} \
            1 {"value" 20 "default" 20} \
            2 {"value" 20 "default" 20} \
            3 {"value" 20 "default" 20} \
            } \
        frame { \
            0 {"value" 0 "default" 0} \
            1 {"value" 0 "default" 0} \
            2 {"value" 0 "default" 0} \
            3 {"value" 0 "default" 0} \
            } \
        jpegquality { \
            0 {"value" 80 "default" 80} \
            1 {"value" 80 "default" 80} \
            2 {"value" 80 "default" 80} \
            3 {"value" 80 "default" 80} \
            } \
        showmask { \
            0 {"value" "off" "default" "off"} \
            1 {"value" "off" "default" "off"} \
            2 {"value" "off" "default" "off"} \
            3 {"value" "off" "default" "off"} \
            } \
        imageupdates { \
            0 {"value" 1 "default" 1} \
            12 {"value" 4 "default" 4} \
            } \
        ]
    # }}} variable "appsettings"

    variable N_ROWS 28
    variable N_COLS 16
    variable N_VORN 14; # Anzahl der Zeilen, die nach vorn gehören. (Berechnung front_rear)
    variable N_UPDATE_INT 4; # Für jedes N-te integrierte Bild ein JPEG-Update senden
    variable N_UPDATE_LIVE 1; # Während der Aufzeichnung nur jedes N-te Livebild als JPEG konvertieren
    variable S_SPEEDCOUNT 10.0; # Zeit für die Geschwindigkeitsmessung in s

    variable recording 0
    variable cop_overload; # Überlastung bei der Berechnung des center of pressure aufgetreten
    variable rec_after_id ""
    variable n_integrated
    variable n_live 0
    variable speedcounting 0; # Zählung für die Geschwindigkeitsmessung läuft
    variable n_speedcount; # Zähler für die Geschwindigkeitsmessung 
    variable channel 1; # Der aktive Kanal 
    variable current_max 0; # in %
    variable started 0; # Die App wurde gestartet.
    variable rotationsthread; # Thread Id
    variable libdir
    variable logging_cop 0; # center of pressure wird gerade von thread geloggt
    variable session_id 0; # id der aktuellen Sitzung (Schlüssel in der lokalen Datenbank), 0 <=> Es gibt keine aktuelle Sitzung.


    # Die Daten des aktuellen Clients
    namespace eval client {; #{{{
        variable N_ROWS [set [namespace parent]::N_ROWS]
        variable N_COLS [set [namespace parent]::N_COLS]

        ### Einstellungen und Daten für die einzelnen Bilder
        # Livebild
        namespace eval live {
            variable N_ROWS [set [namespace parent]::N_ROWS]
            variable N_COLS [set [namespace parent]::N_COLS]

            #??? set maskvalues [dict get [set [namespace parent [namespace parent]]::masks] 0 values]
            set maskvalues [lrepeat [expr ${N_ROWS}*${N_COLS}] 0]
            set jpegoptions [dict create]
            set showmask 1
        }; # live

        # Integration 1
        namespace eval int1 {
            variable N_ROWS [set [namespace parent]::N_ROWS]
            variable N_COLS [set [namespace parent]::N_COLS]

            # Integriertes Bild
            set imagedata [dict create values [lrepeat [expr ${N_ROWS}*${N_COLS}] 0] n_rows ${N_ROWS} n_cols ${N_COLS} max 0] 
            dict set imagedata mask [lrepeat [expr ${N_ROWS}*${N_COLS}] 0]
            # Aktuelle integrierte Zwischenwerte (Gleitkomma)
            set intvalues [lrepeat [expr ${N_ROWS}*${N_COLS}] 0.0]
            set jpegoptions [dict create]
            set showmask 0
            set finished 0; # Timestamp für das Ende der Aufzeichnung [clock seconds]
            set results ""; # JSON
            set recording_id 0; # Die Aufzeichnung wurde gespeichert
        }; # int1

        # Integration 2
        namespace eval int2 {
            variable N_ROWS [set [namespace parent]::N_ROWS]
            variable N_COLS [set [namespace parent]::N_COLS]

            # Integriertes Bild
            set imagedata [dict create values [lrepeat [expr ${N_ROWS}*${N_COLS}] 0] n_rows ${N_ROWS} n_cols ${N_COLS} max 0] 
            dict set imagedata mask [lrepeat [expr ${N_ROWS}*${N_COLS}] 0]
            # Aktuelle integrierte Zwischenwerte (Gleitkomma)
            set intvalues [lrepeat [expr ${N_ROWS}*${N_COLS}] 0.0]
            set jpegoptions [dict create]
            set showmask 0
            set finished 0; # Timestamp für das Ende der Aufzeichnung [clock seconds]
            set results ""; # JSON
            set recording_id 0; # Die Aufzeichnung wurde gespeichert
        }; # int2

        # Integration (sonstige, gespeicherte Bilder)
        namespace eval int3 {
            # Optionen
            set jpegoptions [dict create]
            set showmask 0
        }; # int
        #}}}
    }; # Die Daten des aktuellen Clients


        # Client-Daten zurücksetzen
        # 0 live, 1, 2 integriertes Druckbild
        # Keine Angabe => alle
        # @param [0] [1] [2]
        proc resetData {args} {; #{{{
            variable N_ROWS
            variable N_COLS

            if {[llength $args] == 0} {
                set args [list 0 1 2]
            }
            if {0 in $args} {
                set [namespace current]::client::live::maskvalues [lrepeat [expr ${N_ROWS}*${N_COLS}] 0]
                set [namespace current]::client::live::jpegoptions [dict create]
                set [namespace current]::client::live::showmask 1
                set args [lrange $args 1 end]
            }
            foreach int $args {
                set int "int$int"
                set [namespace current]::client::${int}::imagedata [dict create values [lrepeat [expr ${N_ROWS}*${N_COLS}] 0] n_rows ${N_ROWS} n_cols ${N_COLS} max 0] 
                dict set [namespace current]::client::${int}::imagedata mask [lrepeat [expr ${N_ROWS}*${N_COLS}] 0]
                set [namespace current]::client::${int}::intvalues [lrepeat [expr ${N_ROWS}*${N_COLS}] 0.0]
                set [namespace current]::client::${int}::jpegoptions [dict create]
                set [namespace current]::client::${int}::showmask 0
            }
            #}}}
        }; # proc resetData 

	#{{{ rotationsthread (Thread zum Festhalten der Beckenrotation)

	# Die Threadprozeduren
	set threadproc {; #{{{
	
	    # Nächstes Druckzentrum (center of pressure) berechnen und speichern
        # @param sender_id  Thread an den die Fertigmeldung geht
        # @param cmd        Fertigmeldung
	    proc logCop {sender_id cmd druckbild} {
	        ::druckbild::beckenrotation logcop $druckbild $::N_ROWS $::N_COLS
            thread::send -async $sender_id $cmd
	    }
	
	    # Ergebnis der Beckenrotation berechnen und zurückgeben
	    proc getResult {} {
	        return [::druckbild::beckenrotation result]
	    }
        #}}}

	}; # Threadprozeduren
	
    set libdir [file dirname [info script]]
    # "libdir" ist das Verzeichnis DIESES Skripts, nicht das der gesamten Anwendung.
    if {[string equal -length 1 $libdir "."]} {
        set libdir "$env(PWD)[string range $libdir 1 end]"
    }
    append libdir "/lib/[exec uname -m]"

	# Thread zum Festhalten der Beckenrotation während der Integration
	set rotationsthread [thread::create "
            # Wir sind hier im globalen Namespace des Thread-Interpreters.
            load $libdir/libtcl[info tclversion]beckenrotation.so
            thread::send [thread::id] {srvLog rotationsthread Info {loaded: $libdir/libtcl[info tclversion]beckenrotation.so}}
	
	        set ::N_COLS $N_COLS
	        set ::N_ROWS $N_ROWS
	
	    $threadproc
	
	    thread::wait
	"]; # rotationsthread
    srvLog [namespace current] Info "rotationsthread gestartet. id = $rotationsthread"

	#}}} rotationsthread


    # Handler (Callback) für TTYSattel und BTSattel
    # Nimmt die neuesten Druckdaten zwecks Weiterverwendung entgegen.
    # n_rows und n_cols werden hier vertauscht, weil der Controller ein waagerechtes Bild liefert.
    proc handleDBLDValues {values n_cols n_rows} {; #{{{
        variable N_UPDATE_LIVE
        variable N_UPDATE_INT
        variable appsettings
        variable n_live
        variable recording
        variable cop_overload
        variable n_integrated
        variable channel
        variable current_max
        variable rotationsthread 
        variable logging_cop
        variable speedcounting
        variable n_speedcount


        # Neues Maximum bestimmen
        set new_max 0
        set i 0
        foreach value $values {
            if {[lindex $client::live::maskvalues $i]} {; # maskierter Wert
                set value 0
                lset values $i 0
            }
            if {$value > $new_max} {
                set new_max $value
            }
            incr i
        }
        # alles_0 und max wird später noch gebraucht.
        set alles_0 [expr !$new_max]
        set max $new_max
        #   Wert in % bezogen auf 0...32767
        set new_max [expr round(($new_max * 100.0) / 32767.0)]
        set diff [expr abs($new_max - $current_max)]
        if {$diff >= 5 || ($diff > 0 && $new_max == 0)} {
            ::WSServer::disposeServerMessage apps/satteldruckanalyse text "{\"wsevent\": \"max_value\", \"value\": $new_max}"
        }
        set current_max $new_max

        # Livebild mit den eingestellten JPEG-Optionen erzeugen
        if {($n_live % $N_UPDATE_LIVE) == 0 || $current_max == 0} {
            # JPEG-Generierung (falls eingeschaltet) starten
            if {[dict get $appsettings create_jpeg value] == "on"} {
                ::kernel::JPEGSattel::createJPEG sda_live $values $n_rows $n_cols $client::live::jpegoptions
            }
            # Druckbildnormierung (falls eingeschaltet) starten
            if {[dict get $appsettings create_norm value] == "on"} {
                ::kernel::NormSattel::createNorm sda_live $values $n_rows $n_cols
            }
            # (Variable in appsettings ergänzen)
            set n_live 0
        }
        incr n_live

        if {$speedcounting} {
            incr n_speedcount
        }

        if {!$recording} {
            return
        }

        # Integration läuft, falls hier angekommen

        # Werte übernehmen
        #TODO Speichern für Film
        if {$n_integrated == 0} {
            dict set client::int${channel}::imagedata values $values max $max
            set client::int${channel}::intvalues $values
            set intvalues $values
            set n_integrated 1
        } else {; # Werte in Integration übernehmen
            set intvalues [set client::int${channel}::intvalues]
            # Bilder ohne Druck nicht berücksichtigen
            if {!$alles_0} {
                set i 0
                foreach v $values {
                    # In der Tk-Anwendung werden Nullen nie berücksichtigt.
                    #TODO Integrationsverhalten diesbezüglich konfigurierbar machen.
                    set vi [lindex $intvalues $i]
                    lset intvalues $i [expr $vi - $vi/$n_integrated + double([lindex $values $i])/$n_integrated]
                    incr i
                }
                set client::int${channel}::intvalues $intvalues
                incr n_integrated
            }
        }; # Werte in Integration übernehmen
        #srvLog [namespace current] Debug "n_integrated = $n_integrated"

	    # Center of pressure für Beckenrotation berechnen und speichern lassen
        if {$logging_cop} {; # Letzter Aufruf noch nicht beendet.
            if {!$cop_overload} {; # Problem noch nicht gemeldet.
                set cop_overload 1
                srvLog [namespace current]::handleDBLDValues Warn "Overload logging center of pressure"
            }
            # Overload tritt bei 8 Bildern/s nicht auf.
        } else {
            thread::send -async $rotationsthread "::logCop [thread::id] \{set [namespace current]::logging_cop 0\} \{$values\}"
            set logging_cop 1
        }

        # Zwischenausgabe des integrierten Bildes
        # Dazu müssen die integrierten Werte gerundet sein!
        if {$n_integrated % $N_UPDATE_INT == 0} {
            set imagevalues [list]; # Die integrierten Werte als Integer
            set max 0
            foreach vi $intvalues {
                set v [expr int($vi)]
                lappend imagevalues $v
                if {$v > $max} {
                    set max $v
                }
            }
            dict set client::int${channel}::imagedata values $imagevalues
            dict set client::int${channel}::imagedata max $max
            # Vergleichsmodus berücksichtigen
            if {[dict exists [set client::int${channel}::jpegoptions] -maximum]} {; # Vergleichsmodus
                set channel2 [expr $channel == 1 ? 2 : 1]
                set max2 [dict get [set client::int${channel2}::imagedata] max]
                srvLog [namespace current] Debug "$max <=> $max2"
                if {$max2 > $max} {
                    set max $max2
                }
                dict set client::int${channel}::jpegoptions -maximum $max
                dict set client::int${channel2}::jpegoptions -maximum $max
                # Bild für $channel2 ebenfalls aktualisieren
                # JPEG-Generierung (falls eingeschaltet) starten
                if {[dict get $appsettings create_jpeg value] == "on"} {
                    ::kernel::JPEGSattel::createJPEG sda_int$channel2 [dict get [set client::int${channel2}::imagedata] values] $n_rows $n_cols "[set client::int${channel2}::jpegoptions]"
                }
                # Druckbildnormierung (falls eingeschaltet) starten
                if {[dict get $appsettings create_norm value] == "on"} {
                    ::kernel::NormSattel::createNorm sda_int$channel2 [dict get [set client::int${channel2}::imagedata] values] $n_rows $n_cols "[dict create -maximum $max]"
                }
            }; # if Vergleichsmodus
            # JPEG-Generierung (falls eingeschaltet) starten
            if {[dict get $appsettings create_jpeg value] == "on"} {
                ::kernel::JPEGSattel::createJPEG sda_int$channel $imagevalues $n_rows $n_cols "[set client::int${channel}::jpegoptions]"
            }
            # Druckbildnormierung (falls eingeschaltet) starten
            if {[dict get $appsettings create_norm value] == "on"} {
                ::kernel::NormSattel::createNorm sda_int$channel $imagevalues $n_rows $n_cols "[dict create -maximum $max]"
            }
        }

        #}}}
    }; # proc handleDBLDValues 


    # Bereitstellung eines neuen JPEG-Bildes einer der 3 Bildtypen
    # sda_live sda_int1 sda_int2 sda_int3 an die Anwendung melden.
    proc jpegFinished {imagetype} {; #{{{
        #srvLog [namespace current] Debug "::jpegFinished $imagetype"
        if {"$imagetype" in {sda_live sda_int1 sda_int2 sda_int3}} {
            ::WSServer::disposeServerMessage apps/satteldruckanalyse text "{\"wsevent\": \"$imagetype\"}"
        }
        #}}}
    }; # proc jpegFinished 


    # Normiertes Druckbild ausliefern
    # @param imagetype  Bildtyp
    # @param druckbild  Druckwerte
    # @param n_rows     Anzahl Zeilen
    # @param n_cols     Anzahl Spalten
    proc normFinished {imagetype druckbild n_rows n_cols} {; #{{{
        set values [join $druckbild ","]
        ::WSServer::disposeServerMessage apps/satteldruckanalyse text "{\"wsevent\": \"sattelnorm\", \"imagetype\": \"$imagetype\", \"n_rows\": $n_rows, \"n_cols\": $n_cols, \"values\": \[$values\]}"
        #}}}
    }; # proc normFinished 


    # Die Aufzeichnung wurde beendet.
    # => Letzte Daten berechnen.
    # @param cancel Aufzeichnung abgebrochen => Keine Auswertung
    proc recordingFinished {{cancel 0}} {; #{{{
        variable N_ROWS
        variable N_COLS
        variable N_VORN
        variable appsettings 
        variable recording
        variable rec_after_id
        variable n_integrated
        variable channel
        variable rotationsthread 

        after cancel $rec_after_id
        set rec_after_id ""
        set recording 0
        # Integration in (letztes) Bild umsetzen
        set imagevalues [list]; # Die integrierten Werte als Integer
        set max 0
        foreach vi [set client::int${channel}::intvalues] {
            set v [expr int($vi)]
            lappend imagevalues $v
            if {$v > $max} {
                set max $v
            }
        }
        dict set client::int${channel}::imagedata values $imagevalues
        dict set client::int${channel}::imagedata max $max
        # ... und anzeigen
        # Vergleichsmodus berücksichtigen
        if {[dict exists [set client::int${channel}::jpegoptions] -maximum]} {; # Vergleichsmodus
            set channel2 [expr $channel == 1 ? 2 : 1]
            set max2 [dict get [set client::int${channel2}::imagedata] max]
            if {$max2 > $max} {
                set max $max2
            }
            dict set client::int${channel}::jpegoptions -maximum $max
            dict set client::int${channel2}::jpegoptions -maximum $max
            # Bild für $channel2 ebenfalls aktualisieren
            # JPEG-Generierung (falls eingeschaltet) starten
            if {[dict get $appsettings create_jpeg value] == "on"} {
                ::kernel::JPEGSattel::createJPEG sda_int$channel2 [dict get [set client::int${channel2}::imagedata] values] $N_ROWS $N_COLS "[set client::int${channel2}::jpegoptions]"
            }
            # Druckbildnormierung (falls eingeschaltet) starten
            if {[dict get $appsettings create_norm value] == "on"} {
                ::kernel::NormSattel::createNorm sda_int$channel2 [dict get [set client::int${channel2}::imagedata] values] $n_rows $n_cols "[dict create -maximum $max]"
            }
        }; # if Vergleichsmodus
        # JPEG-Generierung (falls eingeschaltet) starten
        if {[dict get $appsettings create_jpeg value] == "on"} {
            ::kernel::JPEGSattel::createJPEG sda_int$channel $imagevalues $N_ROWS $N_COLS "[set client::int${channel}::jpegoptions]"
        }
        # Druckbildnormierung (falls eingeschaltet) starten
        if {[dict get $appsettings create_norm value] == "on"} {
            ::kernel::NormSattel::createNorm sda_int$channel $imagevalues $N_ROWS $N_COLS "[dict create -maximum $max]"
        }

        if {$cancel} {
            srvLog [namespace current] Info "recording cancelled"
            return
        }

        ## Auswertung
        # Ergebnis für die Beckenrotation holen
        thread::send $rotationsthread ::getResult br
        if {[dict size $br] == 0} {; # Keine Daten gesammelt
            set result [::apps::createJSONError {user 1} record "Insufficient data"]
            srvLog [namespace current] Info $result
            ::WSServer::disposeServerMessage apps/satteldruckanalyse text $result
            return
        }
        srvLog [namespace current] Debug "Beckenrotation: $br"
        set veloscore [::druckbild::veloscore $imagevalues $N_ROWS $N_COLS $client::live::maskvalues]
        srvLog [namespace current] Debug "Veloscore: $veloscore"
        # Summenberechnungen
        set n_vorn $N_VORN
        set n_links [expr $N_COLS / 2]
        set sum_vorn 0
        set sum_hinten 0
        set sum_links 0
        set sum_rechts 0
        set sum 0
        set avg 0
        set max 0
        set n 0; # Anzahl Werte > 0
        set i_row $N_ROWS; # Zeilenindex von vorn
        # Nach vorn/hinten summieren
        # Der Sattel ist von hinten nach vorn in $imagevalues gespeichert.
        for {set i 0} {$i < [llength $imagevalues]} {incr i} {; #{{{ Summen ermitteln
            if {$i % $N_COLS == 0} {
                # Eine Zeile geschafft.
                incr i_row -1
                set i_col 0
            }
            incr i_col
            if {[lindex $client::live::maskvalues $i]} {
                continue
            }
            set wert [lindex $imagevalues $i]
            if {$i_row <= $n_vorn} {
                incr sum_vorn $wert
            } else {
                incr sum_hinten $wert
                # links/rechts nur hinten
                if {$i_col <= $n_links} {
                    incr sum_links $wert
                } else {
                    incr sum_rechts $wert
                }
            }
            if {$wert > $max} {
                set max $wert
            }
            if {$wert > 0} {
                incr n
            }
            #}}}
        }; # Summen ermitteln
        set sum [expr $sum_vorn + $sum_hinten]
        if {$n > 0} {
            set avg [expr $sum / ${n}.0]
        }
        set sum_lr [expr $sum_links + $sum_rechts]
        if {$sum == 0 || $sum_lr == 0} {
            set result [::apps::createJSONError {user 1} record "Insufficient data"]
            srvLog [namespace current] Info $result
            ::WSServer::disposeServerMessage apps/satteldruckanalyse text $result
            return
        }
        # Schwerpunkte
        set analyse_std [::druckbild::analyse std $imagevalues $N_ROWS $N_COLS]
        srvLog [namespace current] Debug "Ergebnis: $analyse_std"

        # Jetzt erst sind wir fertig. Vorher hätte es noch Fehler bei der Auswertung geben können.
        set client::int${channel}::finished [clock seconds]

        #{{{ JSON zusammenbauen => result_data
        set result_data "\"pelvisrotation\": {\"width\": [format {%.1f} [dict get $br width]], \
                                          \"height\": [format {%.1f} [dict get $br height]], \
                                          \"center\": {\"x\": [format {%.1f} [dict get $br center_x]], \"y\": [format {%.1f} [dict get $br center_y]]}, \
                                          \"front\": {\"x\": [format {%.1f} [dict get $br front_x]], \"y\": [format {%.1f} [dict get $br front_y]]}, \
                                          \"right\": {\"x\": [format {%.1f} [dict get $br right_x]], \"y\": [format {%.1f} [dict get $br right_y]]}, \
                                          \"rear\": {\"x\": [format {%.1f} [dict get $br rear_x]], \"y\": [format {%.1f} [dict get $br rear_y]]}, \
                                          \"left\": {\"x\": [format {%.1f} [dict get $br left_x]], \"y\": [format {%.1f} [dict get $br left_y]]}, \
                                          \"c1\": {\"x\": [format {%.1f} [dict get $br c1_x]], \"y\": [format {%.1f} [dict get $br c1_y]]}, \
                                          \"c2\": {\"x\": [format {%.1f} [dict get $br c2_x]], \"y\": [format {%.1f} [dict get $br c2_y]]}, \
                                          \"c3\": {\"x\": [format {%.1f} [dict get $br c3_x]], \"y\": [format {%.1f} [dict get $br c3_y]]}, \
                                          \"c4\": {\"x\": [format {%.1f} [dict get $br c4_x]], \"y\": [format {%.1f} [dict get $br c4_y]]}\
                                         }, \
                     \"veloscore\": [format {%.1f} $veloscore], \
                     \"front_rear\": \"[format "%2.0f" [expr $sum_vorn * 100.0 / $sum]]:[format "%2.0f" [expr $sum_hinten * 100.0 / $sum]]\", \
                     \"left_right\": \"[format "%2.0f" [expr $sum_links * 100.0 / $sum_lr]]:[format "%2.0f" [expr $sum_rechts * 100.0 / $sum_lr]]\", \
                     \"center\": {\"x\": [format {%.1f} [dict get $analyse_std schwerpunkt_x]], \"y\": [format {%.1f} [dict get $analyse_std schwerpunkt_y]]}, \
                     \"center_left\": {\"x\": [format {%.1f} [dict get $analyse_std schwerpunkt1_x]], \"y\": [format {%.1f} [dict get $analyse_std schwerpunkt1_y]]}, \
                     \"center_right\": {\"x\": [format {%.1f} [dict get $analyse_std schwerpunkt2_x]], \"y\": [format {%.1f} [dict get $analyse_std schwerpunkt2_y]]}"
        #}}} JSON zusammengebaut

        # Ergebnis festhalten und an die Clients senden
        set client::int${channel}::results "{$result_data}"
        set wsevent "{\"wsevent\": \"result${channel}\", $result_data}"
        ::WSServer::disposeServerMessage apps/satteldruckanalyse text $wsevent
        srvLog [namespace current] Info $wsevent
        #}}}
    }; # proc recordingFinished 


    # Handler (Callback) für TTY-Treiberereignisse
    # @param change Änderung als wsevent (key/value-Paare)
    proc handleUSBDriverChange {change} {; #{{{
        ::WSServer::disposeServerMessage apps/satteldruckanalyse text [::kvlist2json $change]
        #}}}
    }; # proc handleUSBDriverChange 


    # Handler (Callback) für BT-Treiberereignisse
    # @param change Änderung als JSON
    proc handleBTConnectionChange {json_change} {; #{{{
        srvLog [namespace current]::handleBTConnectionChange Debug $json_change
        # Debug nicht weitergeben
        set dict_change [json::json2dict $json_change]
        set btevent [dict get $dict_change btevent]
        # $json_change in ein wsevent einbetten
        set json_wsevent "{\"wsevent\": \"btevent\", [string range $json_change 1 end-1]}"
        switch $btevent {
            "data" {
                # Datenübernahme auf Bluetooth umschalten
                ::kernel::TTYSattel::removeImageHandler [namespace current]::handleDBLDValues 
                ::kernel::BTSattel::addDataHandler [namespace current]::handleDBLDValues 
            }
            "disconnected" {
                # Datenübernahme auf USB zurückschalten
                ::kernel::BTSattel::removeDataHandler [namespace current]::handleDBLDValues 
                ::kernel::TTYSattel::addImageHandler [namespace current]::handleDBLDValues 
                # Aktuellen TTY Status verteilen, damit die Clients wissen,
                # von dort Daten zu erwarten sind.
                ::kernel::TTYSattel::distributeTTYState
            }
            "ambiguity" {
                # Verteilen an die Clients reicht.
            }
            "status" {
                # Verteilen an die Clients reicht.
            }
            "version" {
                # Verteilen an die Clients reicht.
            }
            "error" {
                # Datenübernahme auf USB zurückschalten
                ::kernel::BTSattel::removeDataHandler [namespace current]::handleDBLDValues 
                ::kernel::TTYSattel::addImageHandler [namespace current]::handleDBLDValues 
                # Aktuellen TTY Status verteilen, damit die Clients wissen,
                # von dort Daten zu erwarten sind.
                ::kernel::TTYSattel::distributeTTYState
            }
            "debug" {
                # Das will der App client nicht wissen.
                return
            }
            default {
                # Das dürfte nie passieren.
                srvLog [namespace current]::handleBTConnectionChange Warn "Unexpected btevent '$btevent' ignored."
                return
            }
        }
        # btevent an die Clients verteilen
        ::WSServer::disposeServerMessage apps/satteldruckanalyse text $json_wsevent
        #}}}
    }; # proc handleBTConnectionChange 


    # Handler (Callback) für Überlastung bei der JPEG-Generierung
    # @param overload   key/value-Liste mit Angaben zur Überlast
    proc handleOverload {overload} {; #{{{
        # (Wenn man das args statt overload benennt, ist $args eine Liste in einer Liste,
        # die erst ausgepackt werden muß.)
        # set msg [dict merge {wsevent overload} {*}$args]
        set msg [dict merge {wsevent overload source jpeg} $overload]
        srvLog [namespace current]::handleOverload Debug $msg
        if {[file exists /sys/class/thermal/thermal_zone0/temp]} {
            # Temperatur einlesen und hinzufügen
            set fd [open /sys/class/thermal/thermal_zone0/temp r]
            set temp [read $fd]
            close $fd
            dict set msg temp [expr $temp / 1000.0]
        }
        if {[file exists /sys/devices/system/cpu/cpu0/cpufreq/scaling_max_freq]} {
            # Maximale CPU-Frequenz einlesen und hinzufügen
            set fd [open /sys/devices/system/cpu/cpu0/cpufreq/scaling_max_freq r]
            set freq_max [read $fd]
            close $fd
            # Frequenz in Mhz statt Hz
            dict set msg freq_max [expr $freq_max / 1000]
        }
        if {[file exists /sys/devices/system/cpu/cpu0/cpufreq/scaling_cur_freq]} {
            # Aktuelle CPU-Frequenz einlesen und hinzufügen
            set fd [open /sys/devices/system/cpu/cpu0/cpufreq/scaling_cur_freq r]
            set freq_cur [read $fd]
            close $fd
            # Frequenz in Mhz statt Hz
            dict set msg freq_cur [expr $freq_cur / 1000]
        }
        ::WSServer::disposeServerMessage apps/satteldruckanalyse text [::kvlist2json $msg]
        #}}}
    }; # proc handleOverload 


    ##{{{ Die Kommandos (Aufruf durch handleCommand)
    #
    # Kommandos geben einen JSON- (wsevent) oder einen Leerstring zurück.
    # Falls es sich um keinen Leerstring handelt, wird die Rückgabe an die Clients verteilt.
    # Fehlermeldungen folgen dem Format von ::apps::createJSONError.
    #
    
    # Aufzeichnung starten/abbrechen
    # Die Aufzeichnung wird nicht gestartet,
    #   wenn bereits eine Aufzeichnung läuft oder
    #   es eine nicht gespeicherte Aufzeichnung gibt.
    #   Im zweiten Fall muß die Aufzeichnung entweder
    #       erst gespeichert werden (recording store ...) oder
    #       erst verworfen werden (recording cancel) oder
    #       es muß die -force-Option verwendet werden.
    # Syntax: record <duration> <cannel>
    # @args
    #       [-force]
    #       Dauer
    #       Kanal [1|2] dflt. 1
    #       oder:
    #       cancel
    # @return   Fehlermeldung als JSON-wsevent oder Leerstring
    proc startRecording {args} {; #{{{
        variable N_ROWS
        variable N_COLS
        variable appsettings 
        variable recording
        variable cop_overload
        variable rec_after_id
        variable n_integrated
        variable channel

        srvLog [namespace current] Debug "Start recording. args: '$args'"
        if {[llength $args] == 1 && [lindex $args 0] == "cancel"} {
            if {$recording} {
                recordingFinished 1
                # recordingFinished meldet den Abbruch im Logfile
                ::WSServer::disposeServerMessage apps/satteldruckanalyse text [::kvlist2json $change]
            }
            return ""
        }
        if {$recording} {
            return [::apps::createJSONError {user 2} record "still recording"]
        }
        # args prüfen
        set force 0
        if {"[lindex $args 0]" == "-force"} {
            set force 1
            set args [lrange $args 1 end]
        }
        set n_args [llength $args]
        if {$n_args in {1 2}} {
            set duration [lindex $args 0]
            if {![string is integer $duration]} {
                return [::apps::createJSONError client record {Invalid duration. Must be integer.}]
            }
            set channel 1; # default
            if {$n_args == 2} {
                set channel [lindex $args 1]
                if {![string is integer $channel] || !($channel in {1 2})} {
                    set channel 1; # vorsichtshalber
                    return [::apps::createJSONError client record {Invalid channel. Must be: 1 or 2}]
                }
            }
        } else {
            return [::apps::createJSONError client record {Invalid args. Must be: record <duration> <channel>}]
        }
        if {!$force} {; # Gibt es ungesicherte Änderungen auf diesem Kanal ?
            if {[set client::int${channel}::finished] > 0 && ![set client::int${channel}::recording_id]} {
                return [::apps::createJSONError {user 3} {session finish} {unstored recording(s)}]
            }
        }
        

        ### Initialisierungen
        srvLog [namespace current] Debug "Initialize recording"
        # Maske übernehmen
        dict set client::int${channel}::imagedata mask $client::live::maskvalues
        if {$client::live::showmask} {
            dict set client::int${channel}::jpegoptions -mask $client::live::maskvalues
        }
        # Werte initialisieren
        dict set client::int${channel}::imagedata values [lrepeat [expr ${N_ROWS}*${N_COLS}] 0]
        set client::int${channel}::intvalues [lrepeat [expr ${N_ROWS}*${N_COLS}] 0.0]
        srvLog [namespace current] Debug "Recording initialized"
        # und anzeigen
        # JPEG-Generierung (falls eingeschaltet) starten
        if {[dict get $appsettings create_jpeg value] == "on"} {
            ::kernel::JPEGSattel::createJPEG sda_int$channel [dict get [set client::int${channel}::imagedata] values] $N_ROWS $N_COLS "[set client::int${channel}::jpegoptions]"
        }
        # Druckbildnormierung (falls eingeschaltet) starten
        if {[dict get $appsettings create_norm value] == "on"} {
            ::kernel::NormSattel::createNorm sda_int$channel [dict get [set client::int${channel}::imagedata] values] $N_ROWS $N_COLS
        }
        set n_integrated 0

        # Zeit *= 1000
        set rec_after_id [after [expr $duration * 1000] [namespace current]::recordingFinished]
        set client::int${channel}::finished 0
        set client::int${channel}::result ""
        set client::int${channel}::recording_id 0
        set recording 1
        set cop_overload 0
        srvLog [namespace current] Info "Recording started"
        return ""
        #}}}
    }; # startRecording


    #{{{ set-Befehle

    # Fehler, die nicht als JSON-String zurückgegeben werden,
    # werden bei den set-Befehlen als Text in einen client set Fehler verpackt.


    # # Kommando "set compare ..."
    # Syntax: set compare on|off
    proc setCompare {args} {; #{{{
        variable N_ROWS
        variable N_COLS
        variable appsettings

        if {[llength $args] != 1} {
            return "Must be: set compare on|off"
        }
        switch [lindex $args 0] {
            "on" {
                foreach c {1 2} {
                    dict set client::int${c}::jpegoptions -maximum [dict get [set client::int${c}::imagedata] max]
                }
            }
            "off" {
                foreach c {1 2} {
                    dict unset client::int${c}::jpegoptions -maximum
                }
            }
            default {
                return "Must be: set compare on|off"
            }
        }
        dict set appsettings compare value [lindex $args 0]
        # Beide Bilder aktualisieren
        if {[dict exists $client::int1::jpegoptions -maximum]} {; # Vergleichsmodus
            set max [dict get $client::int1::imagedata max]
            set max2 [dict get $client::int2::imagedata max]
            if {$max2 > $max} {
                set max $max2
            }
            foreach c {1 2} {
                dict set client::int${c}::jpegoptions -maximum $max
            }
        }
        jpegRefresh 1 2
        return ""
        #}}}
    }; # proc setCompare 


    #TODO Prozedurname anpassen imageRefresh, weil auch normierte Druckwerte geschickt werden.
    # Bilder in den angegebenen Displays aktualisieren
    # Bei Angabe eines anderen Displays als 0, 1 oder 2 passiert nichts.
    # Ebenso passiert nichts, wenn die Anwendung (noch) nicht vollständig gestartet ist.
    # (Unterprogramm von setJPEG)
    # 
    # @param args   [0] [1] [2]
    proc jpegRefresh {args} {; #{{{
        variable N_ROWS
        variable N_COLS
        variable appsettings
        variable started

        if {!$started} {; # Anwendung ist nicht gestartet.
            return
        }
        foreach display $args {
            switch $display {
                0 {
                    # Leeres Livebild schicken
                    set values [lrepeat [expr $N_ROWS*$N_COLS] 0]
                    # JPEG-Generierung (falls eingeschaltet) starten
                    if {[dict get $appsettings create_jpeg value] == "on"} {
                        ::kernel::JPEGSattel::createJPEG sda_live $values $N_ROWS $N_COLS $client::live::jpegoptions
                    }
                    # Druckbildnormierung (falls eingeschaltet) starten
                    if {[dict get $appsettings create_norm value] == "on"} {
                        ::kernel::NormSattel::createNorm sda_live $values $N_ROWS $N_COLS
                    }
                }
                1 {
                    # int1 neu schicken
                    set values [dict get $client::int1::imagedata values]
                    # JPEG-Generierung (falls eingeschaltet) starten
                    if {[dict get $appsettings create_jpeg value] == "on"} {
                        ::kernel::JPEGSattel::createJPEG sda_int1 $values $N_ROWS $N_COLS $client::int1::jpegoptions
                    }
                    # Druckbildnormierung (falls eingeschaltet) starten
                    set normoptions [dict create]
                    if {[dict exists $client::int1::jpegoptions -maximum]} {
                        dict set normoptions -maximum [dict get $client::int1::jpegoptions -maximum]
                    }
                    if {[dict get $appsettings create_norm value] == "on"} {
                        ::kernel::NormSattel::createNorm sda_int1 $values $N_ROWS $N_COLS $normoptions
                    }
                }
                2 {
                    # int2 neu schicken
                    set values [dict get $client::int2::imagedata values]
                    set normoptions [dict create]
                    if {[dict exists $client::int2::jpegoptions -maximum]} {
                        dict set normoptions -maximum [dict get $client::int2::jpegoptions -maximum]
                    }
                    # JPEG-Generierung (falls eingeschaltet) starten
                    if {[dict get $appsettings create_jpeg value] == "on"} {
                        ::kernel::JPEGSattel::createJPEG sda_int2 $values $N_ROWS $N_COLS $client::int2::jpegoptions
                    }
                    # Druckbildnormierung (falls eingeschaltet) starten
                    if {[dict get $appsettings create_norm value] == "on"} {
                        ::kernel::NormSattel::createNorm sda_int2 $values $N_ROWS $N_COLS $normoptions
                    }
                }
            }
        }
        #}}}
    }; # proc jpegRefresh 


    # Kommando "set jpeg ..."
    # Syntax: set jpeg ...
    #       ... colorcontrast 0...7
    #       ... bgcolor #%02x%02x%02x
    #       ... 0|1|2|3 grid no|dotted|solid
    #       ... 0|1|2|3 resolution 1...???
    #       ... 0|1|2|3 frame 0...???
    #       ... 0|1|2|3 quality 0...100
    #       ... 0|1|2|3 mask show|hide
    # @param args   s. Syntax
    # @return    Fehlermeldung oder Leerstring
    proc setJPEG {args} {; #{{{
        variable appsettings

        if {[llength $args] < 2} {
            return {Must be: set jpeg colorcontrast|bgcolor|0...3 ...}
        }
        switch -regexp [lindex $args 0] {
            "colorcontrast" {
                set colorcontrast [lindex $args 1]
                if {[regexp {^[0-7]$} $colorcontrast]} {
                    ::DBLD2IMG::setGlobalJPEG colorcontrast $colorcontrast
                    dict set appsettings contrast value $colorcontrast
                    jpegRefresh 0 1 2
                    return
                } else {
                    return "Must be: set jpeg colorcontrast 0...7"
                }
            }
            "bgcolor" {
                set color [lindex $args 1]
                if {[regexp {^#[0-9A-Fa-f]{6}$} $color]} {
                    ::DBLD2IMG::setGlobalJPEG bgcolor $color
                    dict set appsettings bgcolor value $color
                    jpegRefresh 0 1 2
                    return
                } else {
                    return "Must be: set jpeg bgcolor #%02x%02x%02x"
                }
            }
            {[0-3]} {
                set display [lindex $args 0]
                if {[llength $args] < 3} {
                    return "Must be: set jpeg $display grid|mask|resolution|frame ..."
                }
                set value [lindex $args 2]
                switch [lindex $args 1] {
                    "grid" {
                        if {!($value in {no dotted solid})} {
                            return "Must be: set jpeg $display grid no|dotted|solid"
                        }
                        if {$display == 0} {
                            dict set client::live::jpegoptions -grid $value
                        } else {
                            dict set client::int${display}::jpegoptions -grid $value
                        }
                        dict set appsettings grid $display value $value
                        jpegRefresh $display
                        return
                    }
                    "mask" {
                        switch $value {
                            show {
                                set value 1
                            }
                            hide {
                                set value 0
                            }
                            default {
                                return "Must be: set jpeg $display mask show|hide"
                            }
                        }
                        if {$display == 0} {
                            set client::live::showmask $value
                            if {$value} {
                                dict set client::live::jpegoptions -mask $client::live::maskvalues
                            } else {
                                dict unset client::live::jpegoptions -mask
                            }
                        } else {
                            set client::int${display}::showmask $value
                            if {$value} {
                                dict set client::int${display}::jpegoptions -mask [dict get [set client::int${display}::imagedata] mask]
                            } else {
                                dict unset client::int${display}::jpegoptions -mask
                            }
                        }
                        if {$value} {
                            set value "on"
                        } else {
                            set value "off"
                        }
                        dict set appsettings showmask $display value $value
                        jpegRefresh $display
                        return
                    }
                    "resolution" {
                        if {[string is integer -strict $value]} {
                            if {$value >= 1} {
                                if {$display == 0} {
                                    dict set client::live::jpegoptions -res $value
                                } else {
                                    dict set client::int${display}::jpegoptions -res $value
                                }
                                dict set appsettings resolution $display value $value
                                jpegRefresh $display
                                return
                            }
                        }
                        return "Must be: set jpeg $display resolution 1 ..."
                    }
                    "frame" {
                        if {[string is integer -strict $value]} {
                            if {$value >= 0} {
                                if {$display == 0} {
                                    dict set client::live::jpegoptions -frame $value
                                } else {
                                    dict set client::int${display}::jpegoptions -frame $value
                                }
                                dict set appsettings frame $display value $value
                                jpegRefresh $display
                                return
                            }
                        }
                        return "Must be: set jpeg $display frame 0 ..."
                    }
                    "quality" {
                        if {[string is integer -strict $value]} {
                            if {0 <= $value && $value <= 100} {
                                if {$display == 0} {
                                    dict set client::live::jpegoptions -quality $value
                                } else {
                                    dict set client::int${display}::jpegoptions -quality $value
                                }
                                dict set appsettings jpegquality $display value $value
                                jpegRefresh $display
                                return
                            }
                        }
                        return "Must be: set jpeg $display quality 0 ... 100"
                    }
                    default {
                        return "Must be: set jpeg $display grid|mask|resolution|frame ..."
                    }
                }
                return ""
            }
            default {
                return {Must be: set jpeg colorcontrast|bgcolor|{0...2 grid|mask|resolution|frame ...}}
            }
        }; # switch jpeg-Option
        #}}}
    }; # Kommando "set jpeg ..."


    # Maskenauswahl und Speichern
    # Syntax: set mask {<name> | [-store [<name>]] <ziffern>}
    # @param args   s. Syntax
    proc setMask {args} {; #{{{
        variable N_ROWS
        variable N_COLS
        variable appsettings

        set maskname "custom"
        set store 0
        if {[lindex $args 0] == "-store"} {
            set store 1
            set args [lrange $args 1 end]
        }
        if {[llength $args] == 0} {; # Syntaxfehler
            return "Insufficient arguments. Must be mask name or mask values."
        }
        if {[regexp {^[01]$} [lindex $args 0]]} {; # Neue Werte für die individuelle Maske
            # Werte werden immer gespeichert.
            set store 1
        } else {; # Dann muß es der Maskenname sein.
            set maskname [lindex $args 0]
            if {![string is alnum $maskname]} {
                return "Mask name must be of alphanumeric characters only."
            }
            if {[string is digit -strict [string index $maskname 0]]} {
                return "Mask name must not begin with a digit."
            }
            set args [lrange $args 1 end]
            if {$store && [llength $args] == 0} {
                return "Insufficient arguments. Must have mask values."
            }
        }
        if {$store} {; #{{{ Neue Werte speichern
            # Die verbleibenden Argumente sind die neuen Werte.
            set newmask $args
            # Gültigkeit prüfen
            if {[llength $newmask] != [expr ${N_COLS}*${N_ROWS}]} {
                return "Invalid mask size ([llength $newmask]). Must be [expr ${N_COLS}*${N_ROWS}]."
            }
            foreach maskpoint $newmask {
                if {!($maskpoint in {0 1})} {
                    return "Invalid mask value. Must be 0 or 1."
                }
            }
            srvLog [namespace current] Debug "Mask is valid"
            if {"$maskname" in {nomask standard large}} {
                return "Mask '$maskname' must not be overwitten."
            }
            # Die Maskenwerte von $newmask werden gespiegelt. => $mask2
            set mask2 [list]
            for {set i_row [expr $N_ROWS - 1]} {$i_row >= 0} {incr i_row -1} {
                for {set i_col 0; set i_val [expr $i_row * $N_COLS]} {$i_col < $N_COLS} {incr i_col; incr i_val} {
                    lappend mask2 [lindex $newmask $i_val]
                }
            }
            srvLog [namespace current] Debug "Mask mirrored: $mask2"
            # Maske unter $maskname speichern (neu anlegen oder aktualisieren)
            # Existiert die Maske ?
            set sql "SELECT mask_id FROM masks WHERE name = '$maskname'"
            srvLog [namespace current]::setMask Debug $sql
            set mask_ids [$::kernel::DB::clients allrows $sql]
            if {[llength $mask_ids] == 0} {
                # nein => INSERT
                srvLog [namespace current]::setMask Debug "Mask '$maskname' doesn't exist."
                if {[catch {
                    $::kernel::DB::clients begintransaction
                    set mask_id [::kernel::DB::nextId masks]
                    set sql "INSERT INTO masks(mask_id, name, n_rows, n_cols, mask) VALUES($mask_id, '$maskname', $N_ROWS, $N_COLS, '$mask2')"
                    srvLog [namespace current]::setMask Debug $sql
                    $::kernel::DB::clients allrows $sql
                    $::kernel::DB::clients commit
                } msg]} {
                    $::kernel::DB::clients rollback
                    srvLog [namespace current]::setMask Error $msg
                    return [::apps::createJSONError internal {set mask} {Database error (s. log)}]
                }
            } else {
                # ja => UPDATE
                srvLog [namespace current]::setMask Debug "Mask '$maskname' exists."
                if {[catch {
                    $::kernel::DB::clients begintransaction
                    set mask_id [::kernel::DB::nextId masks]
                    set sql "UPDATE masks SET mask='$mask2' WHERE name='$maskname'"
                    srvLog [namespace current]::setMask Debug $sql
                    $::kernel::DB::clients allrows $sql
                    $::kernel::DB::clients commit
                } msg]} {
                    $::kernel::DB::clients rollback
                    srvLog [namespace current]::setMask Error $msg
                    return [::apps::createJSONError internal {set mask} {Database error (s. log)}]
                }
            }
            # Maske in der Anwendung benutzen
            #dict set masks $mask_id values $mask2
            # Maske sofort verwenden
            set client::live::maskvalues $mask2
            #srvLog [namespace current] Debug "showmask $client::live::showmask"
            if {$client::live::showmask} {
                srvLog [namespace current]::setMask Debug "Mask will be shown."
                dict set client::live::jpegoptions -mask $mask2
                jpegRefresh 0
            }
            #}}}
        } else {; #{{{ gespeicherte Maske $maskname holen
            set sql "SELECT mask FROM masks WHERE name = '$maskname'"
            srvLog [namespace current]::setMask Debug $sql
            set masks [$::kernel::DB::clients allrows $sql]
            if {[llength $masks] == 0} {
                return "Maskname '$maskname' unknown"
            }
            set maskvalues [dict get [lindex $masks 0] mask]
            set client::live::maskvalues $maskvalues
            if {$client::live::showmask} {
                srvLog [namespace current]::setMask Debug "Mask will be shown."
                dict set client::live::jpegoptions -mask $maskvalues
                jpegRefresh 0
            }
            #}}}
        }
        dict set appsettings maskname value $maskname
        return ""
        #}}}
    }; # proc setMask 


    # Clienteinstellungen speichern
    # (Es handelt sich um Einstellungen, die nur für die client GUI relevant sind
    #  und von dieser definiert werden.)
    # Syntax: set clientsettings <clientsettings>
    # @param clientsettings JSON string
    proc setClientsettings {clientsettings} {; #{{{
        #TODO Gültiges JSON Stringobjekt ? (Fehler werden spätestens beim INSERT bemerkt.)
        return [::kernel::DB::setClientsettings "satteldruckanalyse" $clientsettings]
        #}}}
    }; # proc setClientsettings 

    #}}} set-Befehle


    #{{{ get-Befehle

    # Fehler, die nicht als JSON-String zurückgegeben werden,
    # werden bei den set-Befehlen als Text in einen client get Fehler verpackt.

    # Send mask <name> as wsevent "mask" to client
    # Kommando: get mask <name>
    proc getMask {args} {; #{{{
        variable N_ROWS
        variable N_COLS

        if {[llength $args] != 1} {
            return "Must be: get mask <name>"
        }
        set maskname [lindex $args 0]
        set sql "SELECT mask FROM masks WHERE name = '$maskname'"
        srvLog [namespace current]::getMask Debug $sql
        if {[catch {
            set masks [$::kernel::DB::clients allrows $sql]
        } msg]} {
            srvLog [namespace current]::getMask Error $msg
            return [::apps::createJSONError internal {recording store} {Database error (s. log)}]
        }
        if {[llength $masks] == 0} {
            return "Maskname '$maskname' unknown"
        }
        set maskvalues [dict get [lindex $masks 0] mask]
        srvLog [namespace current]::getMask Debug "Mask: $maskvalues"
        # Das muß noch gespiegelt werden. => mask2
        set mask2 [list]
        for {set i_row [expr $N_ROWS - 1]} {$i_row >= 0} {incr i_row -1} {
            for {set i_col 0; set i_val [expr $i_row * $N_COLS]} {$i_col < $N_COLS} {incr i_col; incr i_val} {
                lappend mask2 [lindex $maskvalues $i_val]
            }
        }
        srvLog [namespace current]::getMask Debug "Mask mirrored: $mask2"
        set wsevent "{\"wsevent\": \"mask\", \"name\": \"$maskname\", \"values\": \[[string map {{ } ,} $mask2]\]}"
        ::WSServer::disposeServerMessage apps/satteldruckanalyse text $wsevent
        return ""
        #}}}
    }; # proc getMask 


    # Schickt das Abfrageformular für diese Anwendung an den Client
    # Kommando: get appquestions [<lang>]
    proc getAppquestions {args} {; #{{{
        switch [llength &args] {
            0 {
                set lang "de"
            }
            1 {
                set lang [lindex $args 0]
            }
            default {
                return "Must be: get questions \[<lang>\]"
            }
        }
        set questions [::kernel::DB::getAppquestions "satteldruckanalyse" $lang]
        set wsevent "{\"wsevent\": \"appquestions\", \"appquestions\": $questions}"
        ::WSServer::disposeServerMessage apps/satteldruckanalyse text $wsevent
        return ""
        #}}}
    }; # proc getAppquestions 


    # Aktuelle Einstellungen der App an den Client senden
    # Kommando: get appsettings
    proc getAppsettings {} {; #{{{
        variable appsettings

        # appsettings ohne defaults ausliefern
        set as2store $appsettings
        dict for {outer value} $as2store {
            if {[dict exists $value "default"]} {
                dict unset as2store $outer "default"
            } else {
                dict for {inner dummy} $value {
                    dict unset as2store $outer $inner "default"
                }
            }
        }
        set jsonsettings ""
        dict for {key value} $as2store {
            if {"$jsonsettings" != ""} {
                append jsonsettings ", "
            }
            append jsonsettings [::kernel::DB::dict2JSONsetting $key $value]
        }
        set wsevent "{\"wsevent\": \"appsettings\", \"appsettings\": {$jsonsettings}}"
        ::WSServer::disposeServerMessage apps/satteldruckanalyse text $wsevent
        return ""
        #}}}
    }; # proc getAppsettings 
    

    #TODO Dok
    proc getClientsettings {} {
        set clientsettings [::kernel::DB::getClientsettings "satteldruckanalyse"]
        set wsevent "{\"wsevent\": \"clientsettings\", \"clientsettings\": $clientsettings}"
        ::WSServer::disposeServerMessage apps/satteldruckanalyse text $wsevent
        return ""
    }

    #}}} get-Befehle


    # Numerische Konfigurationseinstellungen
    # @param args   Liste mit 1x was? und wert
    proc configValue {args} {; #{{{
        variable N_UPDATE_LIVE
        variable N_UPDATE_INT
        variable appsettings

        if {[llength $args] != 2} {
            return [::apps::createJSONError client config "Must be: config <what> <value>"]
        }
        set value [lindex $args 1]
        if {![string is integer -strict $value]} {
            return [::apps::createJSONError client config "Must be: config <what> <integer>"]
        }
        switch [lindex $args 0] {
            "n_update_live" {
                if {$value < 1} {
                    return [::apps::createJSONError client config "Must be: config n_update_live 1 ..."]
                }
                set N_UPDATE_LIVE $value
                dict set appsettings imageupdates 0 value $value
            }
            "n_update_int" {
                if {$value < 1} {
                    return [::apps::createJSONError client config "Must be: config n_update_int 1 ..."]
                }
                set N_UPDATE_INT $value
                dict set appsettings imageupdates 12 value $value
            }
            default {
                return [::apps::createJSONError client config "Must be: config n_update_live|n_update_int"]
            }
        }
        return ""
        #}}}
    }; # configValue 


    # on/off Konfigurationseinstellungen
    # @param args   Liste mit 1x was? und wert
    proc configSwitch {args} {; #{{{
        variable appsettings

        if {[llength $args] != 2} {
            return [::apps::createJSONError client config "Must be: switch <what> on|off"]
        }
        set value [lindex $args 1]
        if {!($value in {on off})} {
            return [::apps::createJSONError client config "Must be: switch <what> on|off"]
        }
        switch [lindex $args 0] {
            "create_jpeg" {
                dict set appsettings create_jpeg value $value
            }
            "create_norm" {
                dict set appsettings create_norm value $value
            }
            default {
                return [::apps::createJSONError client config "Must be: switch create_jpeg|create_norm on|off"]
            }
        }
        return ""
        #}}}
    }; # configSwitch 


    # Callback für Statusabfrage "imagespeed"
    proc cbImagespeed {} {; #{{{
        variable S_SPEEDCOUNT
        variable speedcounting
        variable n_speedcount

        set speedcounting 0
        set speed [expr $n_speedcount / $S_SPEEDCOUNT]
        ::WSServer::disposeServerMessage apps/satteldruckanalyse text [::kvlist2json "wsevent status imagespeed [format {%.1f} $speed]"]
        #}}}
    }; # proc cbImagespeed 


    # Statusabfragen
    proc status {args} {; #{{{
        variable S_SPEEDCOUNT
        variable speedcounting
        variable n_speedcount

        if {[llength $args] != 1} {
            return [::apps::createJSONError client status "Must be: status <what>"]
        }
        set what [lindex $args 0]
        switch -regexp $what {
            {^imagespeed$} {
                set n_speedcount 0
                set speedcounting 1
                after [expr round($S_SPEEDCOUNT * 1000)] [namespace current]::cbImagespeed
            }
            {^temp$} {
                if {[file exists /sys/class/thermal/thermal_zone0/temp]} {
                    # Temperatur einlesen und hinzufügen
                    set fd [open /sys/class/thermal/thermal_zone0/temp r]
                    set temp [read $fd]
                    close $fd
                    set temp [expr $temp / 1000.0]
                    ::WSServer::disposeServerMessage apps/satteldruckanalyse text [::kvlist2json "wsevent status temp [format {%.1f} $temp]"]
                } else {
                    return [::apps::createJSONError internal status "temp not available."]
                }
            }
            {^channel[12]$} {
                set channel [namespace current]::client::int[string index $what end]
                if {[set ${channel}::finished]} {
                    set finished "true"
                } else {
                    set finished "false"
                }
                # Status als JSON Objekt:
                set status "{\"finished\":\"$finished\", \"recording_id\":[set ${channel}::recording_id]}"
                ::WSServer::disposeServerMessage apps/satteldruckanalyse text "{\"wsevent\":\"status\", \"$what\": $status}"
            }
            default {
                return [::apps::createJSONError client status {Must be: status imagespeed|temp|channel{1|2}}]
            }
        }
        return ""
        #}}}
    }; # proc status 


    ##{{{ Bluetooth Kommandos

    # Auf Bluetooth umschalten
    # btconnect [<device_name>]
    # @return   Fehlermeldung als JSON-wsevent oder Leerstring
    proc btconnect {args} {; #{{{
        switch [llength $args] {
            0 {
                ::kernel::BTSattel::tryConnect
            }
            1 {
                ::kernel::BTSattel::tryConnect [lindex $args 0]
            }
            default {
                return [::apps::createJSONError client btconnect {Invalid args. Must be btconnect [<device_name>]}]
            }
        }
        return ""
        #}}}
    }; # proc btconnect 


    # Bluetooth Verbindung beenden
    # btdisconnect
    proc btdisconnect {args} {; #{{{
        ::kernel::BTSattel::disconnect
        return ""
        #}}}
    }; # proc btdisconnect 


    # Bluetooth Geträt(e) entfernen
    # btremove [<device_name>]
    proc btremove {args} {; #{{{
        if {[llength $args] > 1} {
            return [::apps::createJSONError client btremove {Invalid args. Must be btremove [<device_name>]}]
        }
        if {[::kernel::BTSattel::removeDevices [lindex $args 0]]} {
            return ""
        }
        return [::apps::createJSONError internal btremove {Couldn't remove device(s). (s. log)}]
        #}}}
    }; # proc btremove 


    # Version des Bluetooth Agenten abfragen
    proc btversion {args} {; #{{{
        ::kernel::BTSattel::agentVersion
        return ""
        #}}}
    }; # proc btversion 

    ##}}} Bluetooth Kommandos
    

    #{{{ Kommandos mit Datenbank-Interaktion

    # SQL-Befehl in vlbclients ausführen
    proc execSQL {sql} {; #{{{
        srvLog [namespace current]::execSQL Info "$sql"
        if {[catch {
            set rows [$::kernel::DB::clients allrows $sql]
        } msg]} {
            srvLog [namespace current]::exexSQL Error $msg
            return [::apps::createJSONError internal {recording store} {Database error (s. log)}]
        }
        set comma ""
        set json_rows "\["
        foreach row $rows {
            append json_rows "$comma{"
            set comma2 ""
            dict for {column value} $row {
                append json_rows "$comma2\"$column\":\"[string map {\" \\" \n \\n} $value]\""
                set comma2 ", "
            }
            append json_rows "}"
            set comma ", "
        }
        append json_rows "\]"
        # Zeilen in ein wsevent einbetten und wegschicken.
        set wsevent "{\"wsevent\":\"sqlresult\", \"rows\":$json_rows}"
        ::WSServer::disposeServerMessage apps/satteldruckanalyse text $wsevent
        return ""
        #}}}
    }; # proc execSQL 

    # (Beendete) Aufzeichnung verwerfen
    # Syntax: recording cancel [1|2]
    # @param channel    Kanal, dessen Aufzeichnung verworfen werden soll
    proc cancelRecording {{channel {} }} {; #{{{
        srvLog [namespace current]::cancelRecording Debug "channel=$channel"
        if {"$channel" == ""} {
            set channels {1 2}
        } else {
            if {$channel ni {1 2}} {
                return [::apps::createJSONError client {retrieve recording} {Invalid args. Must be recording cancel [1|2]}]
            }
            set channels $channel
        }
        foreach channel $channels {
            if {[set client::int${channel}::recording_id] || ![set client::int${channel}::finished]} {; # bereits gespeicherte oder nicht beemdete Aufzeichnung
                srvLog [namespace current]::cancelRecording Debug "Recording $channel not finished or already stored."
                continue
            }
            set client::int${channel}::finished 0
            set client::int${channel}::recording_id 0
        }
        srvLog [namespace current]::cancelRecording Info "Recording $channel cancelled."
        return ""
        #}}}
    }; # proc cancelRecording 


    # Eine Aufzeichnung speichern
    # Syntax: recording store 1|2 [<infos>]
    # @param channel    Kanal, dessen Aufzeichnung gespeichert werden soll
    # @param infos      JSON String mit den Schlüsseln product product_label notes
    proc storeRecording {channel {infos ""}} {; #{{{
        variable session_id
        variable N_ROWS
        variable N_COLS

        srvLog [namespace current]::storeRecording Debug "channel=$channel infos=\"$infos\""
        if {"$infos" == ""} {
            set dict_infos [dict create]
        } else {
            if {[catch {set dict_infos [::json::json2dict $infos]} msg]} {
                return [::apps::createJSONError client {recording store} "Invalid json: $infos ($msg)"]
            }
        }
        if {[set client::int${channel}::recording_id]} {; # bereits gespeicherte Aufzeichnung
            # Es werden höchstens die Zusatzdaten product, product_label und notes überschrieben.
            set recording_id [set client::int${channel}::recording_id]
            srvLog [namespace current]::storeRecording Debug "recording_id $recording_id exists => UPDATE"
            if {[dict size $dict_infos] == 0} {
                srvLog [namespace current]::storeRecording Info "recording already store and no infos to update."
                return ""
            }
            set sql "UPDATE recordings SET"
            set comma ""
            # Evtl. product, product_label oder notes geändert
            foreach column {product product_label notes} {
                if {[dict exists $dict_infos $column]} {
                    append sql "$comma ${column}='[string map {' ''} [dict get $dict_infos $column]]'"
                    set comma ","
                }
            }
            append sql " WHERE recording_id = $recording_id"
            srvLog [namespace current]::storeRecording Info "$sql"
            if {[catch {
                $::kernel::DB::clients allrows $sql
            } msg]} {
                srvLog [namespace current]::storeRecording Error $msg
                return [::apps::createJSONError internal {recording store} {Database error (s. log)}]
            }
        } else {; # ungespeicherte Aufzeichnung
            if {$session_id == 0} {
                return [::apps::createJSONError {user 4} {recording store} {No session}]
            }       
            if {![set client::int${channel}::finished]} {
                return [::apps::createJSONError {user 5} {recording store} {No recording finished}]
            }
            srvLog [namespace current]::storeRecording Debug "new recording => INSERT"
            set sql "INSERT INTO recordings (session_id, analysis"
            set values "VALUES ($session_id, 2"
            if {[catch {
                $::kernel::DB::clients begintransaction
                set recording_id [::kernel::DB::nextId recordings]
                append sql ", recording_id"
                append values ", $recording_id"
                append sql ", finished"
                append values ", '[clock format [set client::int${channel}::finished] -format "%Y-%m-%d %H:%M:%S"]'"
                append sql ", n_pressure_rows, n_pressure_cols"
                append values ", $N_ROWS, $N_COLS"
                append sql ", pressure_values"
                append values ", '[dict get [set client::int${channel}::imagedata] values]'"
                append sql ", max_pressure"
                append values ", '[dict get [set client::int${channel}::imagedata] max]'"
                append sql ", mask"
                append values ", '[dict get [set client::int${channel}::imagedata] mask]'"
                append sql ", results"
                append values ", '[set client::int${channel}::results]'"
                foreach column {product product_label notes} {
                    if {[dict exists $dict_infos $column]} {
                        append sql ", $column"
                        append values ", '[string map {' ''} [dict get $dict_infos $column]]'"
                    }
                }
                append sql ") $values)"
                srvLog [namespace current]::storeRecording Info "$sql"
                $::kernel::DB::clients allrows $sql
                $::kernel::DB::clients commit
                set client::int${channel}::recording_id $recording_id
            } msg]} {
                $::kernel::DB::clients rollback
                srvLog [namespace current]::storeRecording Error $msg
                return [::apps::createJSONError internal {recording store} {Database error (s. log)}]
            }
        }
        return ""
        #}}}
    }; # proc storeRecording 


    # Aufzeichnung für die Anzeige in einem Integrationskanal holen 
    # (Unterprogramm zu retrieveRecording)
    # Syntax: recording retrieve <recording_id> {1|2}
    # Es wird ein JPEG-Bild erzeugt und als sda_in1 bzw. sda_int2 bereitgestellt.
    # Im Vergleichsmodus wird außerdem das jeweils andere Bild dem neuen gemeinsamen Maximum entsprechend aktualisiert.
    # Außerdem werden die Ergebnisse abgerufen und als result1 bzw. result2 abgeliefert.
    # Wenn es auf dem angegebenen Kanal eine ungespeicherte Aufzeichnung gibt, wird eine Fehlermeldung zurückgegeben.
    # Die Aufzeichnung muß dann vorher gespeichert oder verworfen werden. (TODO Kommandos dafür)
    # @param recording_id   Datenbank id
    # @param channel        Kanal (s. Syntax)
    proc retrieveRecording12 {recording_id channel} {; #{{{
        variable appsettings

        srvLog [namespace current] Debug "retrieveRecording12 $$recording_id $channel"
        if {$channel ni {1 2}} {
            return [::apps::createJSONError client {retrieve recording} {Invalid args. Must be retrieve recording <id> [1|2]}]
        }
        # Gibt es eine ungespeicherte Aufzeichnung auf diesem Kanal?
        if {([set client::int${channel}::finished] > 0 && ![set client::int${channel}::recording_id])} {
            return [::apps::createJSONError {user 3} {session finish} {unstored recording(s)}]
        }
        set sql "SELECT * FROM recordings WHERE recording_id = $recording_id AND analysis = 2"
        srvLog [namespace current]::retrieveRecording12 Info "$sql"
        if {[catch {set recording [lindex [$::kernel::DB::clients allrows $sql] 0]} msg]} {
            srvLog [namespace current]::retrieveRecording12 Error $msg
            return [::apps::createJSONError internal {retrieve recording} {Database error (s. log)}]
        }
        if {"$recording" == ""} {
            return [::apps::createJSONError internal {retrieve recording} "recording recording_id=$recording_id doesn't exist or is not from saddle pressure analysis"]
        }
        srvLog [namespace current]::retrieveRecording12 Info "recording found. finished [dict get $recording finished]"
        set imagevalues [dict get $recording pressure_values]
        set max 0
        foreach imagevalue $imagevalues {
            if {$max < $imagevalue} {
                set max $imagevalue
            }
        }
        set client::int${channel}::recording_id $recording_id
        dict set client::int${channel}::imagedata values $imagevalues
        dict set client::int${channel}::imagedata mask [dict get $recording mask]
        dict set client::int${channel}::imagedata max $max
        if {[set client::int${channel}::showmask]} {
            dict set client::int${channel}::jpegoptions -mask [dict get $recording mask]
        } else {
            dict unset client::int${channel}::jpegoptions -mask
        }
        # Druckbild anzeigen
        if {[dict exists [set client::int${channel}::jpegoptions] -maximum]} {; # Vergleichsmodus
            set channel2 [expr $channel == 1 ? 2 : 1]
            set max2 [dict get [set client::int${channel2}::imagedata] max]
            if {$max2 > $max} {
                set max $max2
            }
            dict set client::int${channel}::jpegoptions -maximum $max
            dict set client::int${channel2}::jpegoptions -maximum $max
            # Bild für $channel2 ebenfalls aktualisieren
            # JPEG-Generierung (falls eingeschaltet) starten
            if {[dict get $appsettings create_jpeg value] == "on"} {
                ::kernel::JPEGSattel::createJPEG sda_int$channel2 [dict get [set client::int${channel2}::imagedata] values] [dict get $recording n_pressure_rows] [dict get $recording n_pressure_cols] "[set client::int${channel2}::jpegoptions]"
            }
            # Druckbildnormierung (falls eingeschaltet) starten
            if {[dict get $appsettings create_norm value] == "on"} {
                ::kernel::NormSattel::createNorm sda_int$channel2 [dict get [set client::int${channel2}::imagedata] values] [dict get $recording n_pressure_rows] [dict get $recording n_pressure_cols] "[dict create -maximum $max]"
            }
        }; # if Vergleichsmodus
        if {[dict get $appsettings create_jpeg value] == "on"} {
            ::kernel::JPEGSattel::createJPEG sda_int$channel $imagevalues [dict get $recording n_pressure_rows] [dict get $recording n_pressure_cols] "[set client::int${channel}::jpegoptions]"
        }
        # Druckbildnormierung (falls eingeschaltet) starten
        if {[dict get $appsettings create_norm value] == "on"} {
            ::kernel::NormSattel::createNorm sda_int$channel $imagevalues [dict get $recording n_pressure_rows] [dict get $recording n_pressure_cols] "[dict create -maximum $max]"
        }
        # results als wsevent weitergeben
        set results [dict get $recording results]
        if {"[string index $results 0]" == "{" && "[string index $results end]" == "}"} {
            set wsevent "[string range $results 0 0]\"wsevent\": \"result${channel}\""
            foreach column {product product_label notes} {
                if {[dict exists $recording $column]} {
                    append wsevent ", \"$column\": \"[string map {\" \\" \n \\n} [dict get $recording $column]]\""
                }
            }
            append wsevent ", [string range $results 1 end]"
            ::WSServer::disposeServerMessage apps/satteldruckanalyse text $wsevent
        }
        return ""
        #}}}
    }; # proc retrieveRecording12


    # Aufzeichnung für die Anzeige außerhalb der Integrationskanäle holen 
    # (Unterprogramm zu retrieveRecording)
    # Syntax: recording retrieve <recording_id> [3]
    # Es wird ein JPEG-Bild erzeugt und als sda_int3 bereitgestellt.
    # Ohne Kanalangabe erfolgt die Bilderzeugung im "session compare mode".
    # Außerdem werden die Ergebnisse abgerufen und als result3 abgeliefert.
    # @param recording_id   Datenbank id
    # @param scm            session compare mode
    proc retrieveRecording3 {recording_id {scm 1}} {; #{{{
        variable appsettings

        srvLog [namespace current] Debug "retrieveRecording3 $recording_id"
        # Aufzeichnung holen
        set sql "SELECT * FROM recordings WHERE recording_id = $recording_id AND analysis = 2"
        srvLog [namespace current]::retrieveRecording3 Info "$sql"
        if {[catch {set recording [lindex [$::kernel::DB::clients allrows $sql] 0]} msg]} {
            srvLog [namespace current]::retrieveRecording3 Error $msg
            return [::apps::createJSONError internal {retrieve recording} {Database error (s. log)}]
        }
        if {"$recording" == ""} {
            return [::apps::createJSONError internal {retrieve recording} "recording recording_id=$recording_id doesn't exist or is not from saddle pressure analysis"]
        }
        srvLog [namespace current]::retrieveRecording3 Info "recording found. finished [dict get $recording finished]"
        set imagevalues [dict get $recording pressure_values]
        # Maximaldruck der zugehörigen Session bestimmen.
        set sql "SELECT max(max_pressure) AS max FROM recordings WHERE session_id = [dict get $recording session_id]"
        srvLog [namespace current]::retrieveRecording3 Info "$sql"
        if {[catch {set max {*}[$::kernel::DB::clients allrows $sql]} msg]} {
            srvLog [namespace current]::retrieveRecording3 Error $msg
            return [::apps::createJSONError internal {retrieve recording} {Database error (s. log)}]
        }
        set max [dict get $max max]
        srvLog [namespace current]::retrieveRecording3 Info "max: '$max'"
        if {$client::int3::showmask} {
            dict set client::int3::jpegoptions -mask [dict get $recording mask]
        } else {
            dict unset client::int3::jpegoptions -mask
        }
        if {$scm} {
            dict set client::int3::jpegoptions -maximum $max
        } else {
            dict unset client::int3::jpegoptions -maximum
        }
        # JPEG-Generierung (falls eingeschaltet) starten
        if {[dict get $appsettings create_jpeg value] == "on"} {
            ::kernel::JPEGSattel::createJPEG sda_int3 [dict get $recording pressure_values] [dict get $recording n_pressure_rows] [dict get $recording n_pressure_cols] $client::int3::jpegoptions
        }
        # Druckbildnormierung (falls eingeschaltet) starten
        if {[dict get $appsettings create_norm value] == "on"} {
            ::kernel::NormSattel::createNorm sda_int3 [dict get $recording pressure_values] [dict get $recording n_pressure_rows] [dict get $recording n_pressure_cols] [dict create -maximum $max]
        }
        # results als wsevent weitergeben
        set results [dict get $recording results]
        if {"[string index $results 0]" == "{" && "[string index $results end]" == "}"} {
            set wsevent "[string range $results 0 0]\"wsevent\": \"result3\""
            foreach column {product product_label notes} {
                if {[dict exists $recording $column]} {
                    append wsevent ", \"$column\": \"[string map {\" \\" \n \\n} [dict get $recording $column]]\""
                }
            }
            append wsevent ", [string range $results 1 end]"
            ::WSServer::disposeServerMessage apps/satteldruckanalyse text $wsevent
        }
        return ""
        #}}}
    }; # proc retrieveRecording3 


    # Daten einer Aufzeichnung holen und an den Client schicken
    # (s. retrieveRecording12 und retrieveRecording3)
    # Syntax: recording retrieve <recording_id> [1|2|3]
    # @param recording_id   Datenbank id
    # @param channel        Kanal (s. Syntax)
    proc retrieveRecording {recording_id {channel ""}} {; #{{{
        srvLog [namespace current] Debug "retrieveRecording $recording_id $channel"
        if {"$channel" == ""} {
            return [retrieveRecording3 $recording_id]
        } elseif {"$channel" == "3"} {
            return [retrieveRecording3 $recording_id 0]
        }
        return [retrieveRecording12 $recording_id $channel]

        #}}}
    }; # proc retrieveRecording 


    # Kommando "recording" mit Unterkommandos
    # recording store 1|2 [<Freitexte als JSON-String>]
    #           retrieve <recording_id> [1|2]
    #           cancel [1|2]
    proc recording {args} {; #{{{
        srvLog [namespace current]::recording Debug "$args"
        switch [lindex $args 0] {
            "store" {
                # Der JSON-String wird wegen seiner geschweiften Klammern von TCL als ein Listenelement gezählt.
                # Die Klammern verschwinden aber und müssen deshalb wieder hinzugefügt werden.
                if {[llength $args] in {2 3}} {
                    return [storeRecording [lindex $args 1] "{[lindex $args 2]}"]
                }
            }
            "retrieve" {
                if {[llength $args] in {2 3}} {
                    return [retrieveRecording {*}[lrange $args 1 2]]
                }
            }
            "cancel" {
                if {[llength $args] <= 2} {
                    return [cancelRecording [lindex $args 1]]
                }
            }
        }
        return [::apps::createJSONError client recording {Invalid args. Must be recording {store 1|2 [<infos>] | retrieve <recording_id> [1|2] | cancel [1|2]}}]
        #}}}
    }; # proc recording 


    # Kommando "analysis"
    # Das Kommando dient dem Speichern und Lesen von einmaligen Analyseergebnissen einer Session.
    # Syntax:
    #   analysis {<blocks> [<session_id>] | <analyses>}
    #   <blocks>:   JSON-Arraystring mit den codes der abzurufenden Analysen
    #               (sitbones, anamnesis, x1, x2, ...)
    #   <session_id>:   ... der zugehörigen Sitzung
    #   <analyses>: JSON-Objectstring mit den Analysen
    #               
    # Die Unterobjekte der Analysen:
    #   sitbones:       sitzknochenabstand, asymmetrie (s. APPQUESTIONS)
    #   anamnesis:      einsatzbereich, fahrleistung, sitzposition (s. APPQUESTIONS)
    #   x<n>:           Wird von der Clientanwendung definiert.
    # Wenn <JSON-objectstring> angegeben ist, wird gespeichert.
    # Andernfalls werden die durch <blocks> definierten Analysen ausgeliefert.
    # Der Default für <session_id> ist die aktuelle Session.
    # Wenn <session_id> fehlt und keine Session aktiv ist, ist das ein Fehler.
    # @param myargs args NICHT als Liste
    proc analysis {myargs} {; #{{{
        variable session_id

        srvLog [namespace current]::analysis Debug "myargs: $myargs"
        # Das Prozedurargument darf nicht "args" heißen, weil es sonst als Liste (fehl-)interpertiert wird.
        set args $myargs

        # Speichern oder abrufen?
        if {[set char0 "[string index $args 0]"] == "\["} {; # Array
            # => abrufen => session_id erforderlich
            if {[regexp {[1-9][0-9]* *$} $args retrieve_session_id]} {; # Integer am Ende
                # => Das ist die session_id
                set retrieve_session_id [string trim $retrieve_session_id]
                set args [regsub { *[1-9][0-9]* *$} $args ""]
            } else {
                set retrieve_session_id $session_id
            }
        }
        if {[catch {set dict_args [::json::json2dict $args]} msg]} {
            return [::apps::createJSONError client analysis "Invalid JSON: '$args'"]
        }
        srvLog [namespace current]::analysis Debug "dict_args: $dict_args"

        if {$char0 == "\["} {; #{{{ Analysen abrufen
            srvLog [namespace current]::analysis Debug "Retrieve analyses: $args"
            if {$retrieve_session_id == 0} {
                return [::apps::createJSONError client analysis "No active session and no session provided."]
            }
            set analyses [dict create]
            foreach block $dict_args {
                switch -regexp $block {
                    {^sitbones$} {
                        set analysis 1
                    }
                    {^anamnesis$} {
                        set analysis 3
                    }
                    {^x[1-9][0-9]*$} {
                        set analysis [expr 100 + [string range $block 1 end]]
                    }
                    default {
                        return [::apps::createJSONError client analysis "Unknown or invalid analysis: '$block'"]
                    }
                }
                set sql "SELECT results FROM recordings WHERE session_id=$retrieve_session_id AND analysis = $analysis"
                srvLog [namespace current]::analysis Info "$sql"
                set recording [$::kernel::DB::clients allrows $sql]
                if {[llength $recording] == 0} {; # recording existiert nicht.
                    # Es ist ein Fehler, wenn schon die Session nicht existiert.
                    set sql "SELECT exists(SELECT * FROM sessions WHERE session_id=$retrieve_session_id)"
                    srvLog [namespace current]::analysis Info "$sql"
                    set exists [$::kernel::DB::clients allrows -as lists $sql]
                    if {!$exists} {
                        return [::apps::createJSONError client analysis "session_id $retrieve_session_id doesn't exist."]
                    }
                    continue
                }
                dict set analyses $block [dict get [lindex $recording 0] results]
            }
            srvLog [namespace current]::analysis Debug "analyses: $analyses"
            # analyses in JSON umbauen und zurückschicken
            set ljson [list]
            dict for {block results} $analyses {
                lappend ljson "\"$block\":$results"
            }
            srvLog [namespace current]::analysis Debug "ljson: $ljson"
            ::WSServer::disposeServerMessage apps/satteldruckanalyse text "{\"wsevent\": \"analyses\", \"analyses\":{[join $ljson ,]}}"
            #}}}
        } else {; #{{{ Analysen speichern
            srvLog [namespace current]::analysis Debug "Store analyses: $args"
            if {$session_id == 0} {
                return [::apps::createJSONError client analysis "No active session."]
            }

            foreach block [dict keys $dict_args] {; #{{{ Blöcke (Analysen) durchgehen
                srvLog [namespace current]::analysis Debug "Store: '$block'"
                switch -regexp $block {
                    {^sitbones$} {
                        set analysis 1
                    }
                    {^anamnesis$} {
                        set analysis 3
                    }
                    {^x[1-9][0-9]*$} {
                        set analysis [expr 100 + [string range $block 1 end]]
                    }
                    default {
                        return [::apps::createJSONError client analysis "Unknown or invalid analysis: '$block'"]
                    }
                }
                srvLog [namespace current]::analysis Debug "Store analysis for: $block ($analysis)"
                if {[catch {; #{{{ Block mit ermittelter Analysekennung speichern
                    $::kernel::DB::clients begintransaction
                    set sql "SELECT recording_id FROM recordings WHERE analysis=$analysis AND session_id=$session_id"
                    srvLog [namespace current]::analysis Info "$sql"
                    set recording [$::kernel::DB::clients allrows $sql]
                    if {[llength $recording] == 0} {; # recording existiert noch nicht.
                        # => Neue Analyse
                        set recording_id [::kernel::DB::nextId recordings]
                        set sql "INSERT INTO recordings (recording_id, session_id, analysis, finished, results)"
                        append sql " VALUES ($recording_id, $session_id, $analysis, '[clock format [clock seconds] -format "%Y-%m-%d %H:%M:%S"]', json_extract('[string map {' ''} $args]', '\$.$block'))"
                        srvLog [namespace current]::analyse Info "$sql"
                    } else {; # recording existiert
                        # => Analyse ersetzen
                        set recording_id [dict get [lindex $recording 0] recording_id]
                        set sql "UPDATE recordings SET results = json_extract('[string map {' ''} $args]', '\$.$block')"
                        append sql " WHERE recording_id = $recording_id"
                        $::kernel::DB::clients allrows $sql
                    }
                    $::kernel::DB::clients allrows $sql
                    $::kernel::DB::clients commit
                    #}}}
                } msg]} {; # Speichern fehlgeschlagen
                    $::kernel::DB::clients rollback
                    srvLog [namespace current]::analysis Error $msg
                    return [::apps::createJSONError internal {analysis} {Database error (s. log)}]
                }
                #}}}
            }; # foreach block

            #}}}
        }; # else Analyse speichern

        return ""
        #}}}
    }; # proc analysis 


    # Aktuelle Sitzung beenden:
    # Syntax: session finish
    # - Fehler bei nicht gesicherter Aufzeichnung
    #   (vorher speichern oder verwerfen)
    # - Session löschen, wenn es keine Aufzeichnungen gibt.
    # 0 => Aktuelle session_id wird 0, d.h. ("keine Session")
    proc finishSession {} {; #{{{
        variable session_id
        variable recording

        if {$session_id == 0} {
            srvLog [namespace current]::finishSession Debug "session_id=0 => nothing to do"
            return ""
        } else {
            srvLog [namespace current]::finishSession Debug "session_id=$session_id"
        }
        # Nicht wenn eine Aufzeichnung läuft
        if {$recording} {
            return [::apps::createJSONError {user 2} {session finish} {still recording}]
        }
        # Nicht wenn es eine ungespeicherte Aufzeichnung gibt
        if {($client::int1::finished > 0 && !$client::int1::recording_id) || ($client::int2::finished > 0 && !$client::int2::recording_id)} {
            return [::apps::createJSONError {user 3} {session finish} {unstored recording(s)}]
        }
        set n_recordings [$::kernel::DB::clients allrows "SELECT count(*) AS n_recordings FROM recordings WHERE session_id=$session_id"]
        if {[dict get [lindex $n_recordings 0] n_recordings] == 0} {
            $::kernel::DB::clients allrows "DELETE FROM sessions WHERE session_id = $session_id"
        }
        srvLog [namespace current]::finishSession Debug "session session_id=$session_id finished."
        set session_id 0
        return ""
        #}}}
    }; # proc finishSession 


    # Neue Session beginnen
    # (Legt einen neuen Datensatz in der Datenbank an.)
    # Sofern es eine noch laufende Session gibt, wird diese vorher beendet.
    # Die session_id der neu angelegten Sitzung kann mit
    #   SELECT next_id-1 AS session_id FROM nextids WHERE table_name='sessions'
    # die client_id des neu angelegten Kunden mit
    #   SELECT next_id-1 AS client_id FROM nextids WHERE table_name='clients'
    # abgefragt werden.
    # Syntax: session start <client_info>
    # @param client_info    Client für den die Session anzulegen ist.
    #                       entweder als
    #                       <client_id> eines existierenden Kunden
    #                       oder als <client_data>
    #                       JSON String mit email, surname, forename für einen neuen Kunden
    #                           Keiner der Schlüssel ist eine Pflichtangabe.
    #                           Wenn ein leeres JSON-Obekt übergeben wird, kann der Kunde
    #                           noch über "first_contact" wiedergefunden werden.
    # @return Fehler oder Leerstring
    proc startSession {client_info} {; #{{{
        variable session_id

        srvLog [namespace current]::startSession Debug "client_info: $client_info"
        if {"[set result [finishSession]]" != ""} {
            return $result
        }

        if {![string is integer -strict $client_info]} {; #{{{ JSON-String => neuen client anlegen
            if {[catch {
                # client-Daten in dict holen
                set dict_client_info [::json::json2dict "{$client_info}"]
                srvLog [namespace current]::startSession Debug "dict_client_info: $dict_client_info"
            } msg]} {
                srvLog [namespace current]::startSession Error $msg
                #TODO Ist das kein client-error ?
                return [::apps::createJSONError internal {session start} {Invalid JSON string}]
            }

            # client anlegen
            if {[catch {
                $::kernel::DB::clients begintransaction
                set client_id [::kernel::DB::nextId clients]
                set sql "INSERT INTO clients (client_id, first_contact"
                set values "$client_id, '[::kernel::DB::currentTimestamp]'"
                foreach column {email forename surname} {
                    if {[dict exists $dict_client_info $column]} {
                        append sql ", $column"
                        append values ", '[string map {' ''} [dict get $dict_client_info $column]]'"
                    }
                }
                append sql ") VALUES ($values)"
                srvLog [namespace current]::startSession Debug $sql
                $::kernel::DB::clients allrows $sql
                $::kernel::DB::clients commit
                srvLog [namespace current]::startSession Info "Client with client_id $client_id inserted to database."
            } msg]} {
                $::kernel::DB::clients rollback
                set session_id 0
                srvLog [namespace current]::startSession Error $msg
                return [::apps::createJSONError internal {session start} {Database error (s. log)}]
            }
            #}}}
        } else {; # Integer => client_id
            set client_id $client_info
        }

        # Neue session anlegen
        if {[catch {
            $::kernel::DB::clients begintransaction
            set session_id [::kernel::DB::nextId sessions]
            set sql "INSERT INTO sessions (session_id, client_id, date) VALUES ($session_id, $client_id, '[::kernel::DB::currentDate]')"
            srvLog [namespace current]::startSession Debug $sql
            $::kernel::DB::clients allrows $sql
            $::kernel::DB::clients commit
            srvLog [namespace current]::startSession Info "Session started: session_id = $session_id"
        } msg]} {
            $::kernel::DB::clients rollback
            set session_id 0
            srvLog [namespace current]::startSession Error $msg
            return [::apps::createJSONError internal {session start} {Database error (s. log)}]
        }
        return ""
        #}}}
    }; # proc startSession 


    # Frühere Session reaktivieren
    # Beginnt mit retrieveSession
    # @param session_id Datenbank Id
    proc restartSession {session_id} {; #{{{
        # variable session_id

        srvLog [namespace current]::restartSession Debug "session_id: $session_id"
        if {"[set result [retrieveSession $session_id]]" != ""} {
            return $result
        }
        # restart setzen
        set sql "UPDATE sessions SET restart='[::kernel::DB::currentDate]' WHERE session_id=$session_id"
        srvLog [namespace current]::restartSession Debug "$sql"
        if {[catch {
            $::kernel::DB::clients allrows $sql
        } msg]} {
            return [::apps::createJSONError internal {session restart} {Database error (s. log)}]
        }
        set [namespace current]::session_id $session_id
        srvLog [namespace current]::restartSession Info "Session restarted: session_id = $session_id"
        return $result
        #}}}
    }; # proc restartSession 


    # Daten einer Session (1. und letzte Aufzeichnung) holen und an die Clients schicken
    # Beginnt mit finishSession
    proc retrieveSession {session_id} {; #{{{
        variable N_ROWS
        variable N_COLS
        variable appsettings

        srvLog [namespace current] Debug "retrieveSession $session_id"
        if {"[set result [finishSession]]" != ""} {
            return $result
        }
        set sql "SELECT s.session_id, r.recording_id FROM sessions s LEFT JOIN recordings r ON r.session_id=s.session_id WHERE s.session_id=$session_id AND analysis=2 ORDER BY r.finished"
        srvLog [namespace current]::retrieveSession Info "$sql"
        if {[catch {set session [$::kernel::DB::clients allrows $sql]} msg]} {
            srvLog [namespace current]::retrieveSession Error $msg
            return [::apps::createJSONError internal {retrieve session} {Database error (s. log)}]
        }
        if {[llength $session] == 0} {
            return [::apps::createJSONError internal {retrieve session} "session session_id=$session_id doesn't exist"]
        }
        resetData 1 2
        if {[dict exists [lindex $session 0] recording_id]} {
            set result [retrieveRecording [dict get [lindex $session 0] recording_id] 1]
        } else {
            # Leeres Druckbild
            # JPEG-Generierung (falls eingeschaltet) starten
            if {[dict get $appsettings create_jpeg value] == "on"} {
                ::kernel::JPEGSattel::createJPEG sda_int1 [dict get $client::int1::imagedata values] $N_ROWS $N_COLS $client::int1::jpegoptions
            }
            # Druckbildnormierung (falls eingeschaltet) starten
            if {[dict get $appsettings create_norm value] == "on"} {
                ::kernel::NormSattel::createNorm sda_int1 [dict get $client::int1::imagedata values] $N_ROWS $N_COLS
            }
            # Leere results
            ::WSServer::disposeServerMessage apps/satteldruckanalyse text "{\"wsevent\": \"result1\"}"
        }
        srvLog [namespace current]::retrieveSession Debug "llength \$session [llength $session]"
        if {"$result"=="" && [llength $session]>1} {
            set result [retrieveRecording [dict get [lindex $session end] recording_id] 2]
        } else {
            # Leeres Druckbild
            # JPEG-Generierung (falls eingeschaltet) starten
            if {[dict get $appsettings create_jpeg value] == "on"} {
                ::kernel::JPEGSattel::createJPEG sda_int2 [dict get $client::int2::imagedata values] $N_ROWS $N_COLS $client::int2::jpegoptions
            }
            # Druckbildnormierung (falls eingeschaltet) starten
            if {[dict get $appsettings create_norm value] == "on"} {
                ::kernel::NormSattel::createNorm sda_int2 [dict get $client::int2::imagedata values] $N_ROWS $N_COLS
            }
            # Leere results
            ::WSServer::disposeServerMessage apps/satteldruckanalyse text "{\"wsevent\": \"result2\"}"
        }
        return $result
        #}}}
    }; # proc retrieveSession 


    # Kommando "session" mit Unterkommandos
    # session start <client_id>
    #         finish
    #         retrieve <session_id>
    #         restart <session_id>
    proc session {args} {; #{{{
        srvLog [namespace current]::session Debug "$args"
        switch [lindex $args 0] {
            "finish" {
                if {[llength $args] == 1} {
                    return [finishSession]
                }
            }
            "start" {
                if {[llength $args] == 2} {
                    # join verhindert den Verlust der äußeren geschweiften Klammern.
                    return [startSession [join [lrange $args 1 end]]]
                }
            }
            "restart" {
                if {[llength $args] == 2} {
                    return [restartSession [lindex $args 1]]
                }
            }
            "retrieve" {
                if {[llength $args] == 2} {
                    return [retrieveSession [lindex $args 1]]
                }
            }
        }
        return [::apps::createJSONError client retrieve {Invalid args. Must be session {start {<client_id>|<client_data>} | restart <session_id> | finish | retrieve <session_id>}}]
        #}}}
    }; # proc session 

    #}}} Kommandos mit Datenbank-Interaktion

    ##}}} Die Kommandos
    

    # Eingegangenes Kommando an die Kommandoprozedur weiterleiten
    # (Aufruf von custom/wsserver/wsdomains/apps/satteldruckanalyse.tcl)
    # Bei Fehlern geht eine Meldung an die Domäne apps/satteldruckanalyse.
    # @param command    Kommando wie vom Client eingegangen
    proc handleCommand {command} {; #{{{
        variable VERSION

	    srvLog [namespace current] Debug "Command received: $command"
        set commandname [regsub {  *.*$} $command ""]
        set args [regsub {[a-zA-Z][a-zA-Z0-9]* *} $command ""]
        set result ""

        if {[catch {
            switch $commandname {
                "config" {
                    set result [configValue {*}$args]
                }
                "switch" {
                    set result [configSwitch {*}$args]
                }
                "set" {; #{{{
                    if {[llength $args] < 1} {
                        set result "set what?"
                    } else {
                        set what [lindex $args 0]
                        set args [lrange $args 1 end]
                        switch $what {
                            "compare" {
                                set result [setCompare {*}$args]
                            }
                            "jpeg" {
                                set result [setJPEG {*}$args]
                            }
                            "mask" {
                                set result [setMask {*}$args]
                            }
                            "clientsettings" {
                                set result [setClientsettings [join $args]]
                            }
                            default {
                                set result "Must be: set compare|jpeg|mask ..."
                            }
                        }
                    }
                    # results, die nicht als JSON-String zurückgegeben wurden, sind client errors
                    if {"$result"!="" && !([string index $result 0]=="{" && [string index $result end]=="}")} {
                        set result [::apps::createJSONError client "set $what" $result]
                    }
                    #}}}
                }
                "get" {; #{{{
                    if {[llength $args] < 1} {
                        set result "set what?"
                    } else {
                        set what [lindex $args 0]
                        set args [lrange $args 1 end]
                        switch $what {
                            "mask" {
                                set result [getMask {*}$args]
                            }
                            "appquestions" {
                                set result [getAppquestions {*}$args]
                            }
                            "appsettings" {
                                set result [getAppsettings]
                            }
                            "clientsettings" {
                                set result [getClientsettings]
                            }
                            default {
                                set result "Must be: get {mask|appquestions|appsettings|clientsettings} ..."
                            }
                        }
                    }
                    # results, die nicht als JSON-String zurückgegeben wurden, sind client errors
                    if {"$result"!="" && !([string index $result 0]=="{" && [string index $result end]=="}")} {
                        set result [::apps::createJSONError client "get $what"  $result]
                    }
                    #}}}
                }
                "record" {
                    set result [startRecording {*}$args]
                }
                "analysis" {
                    set result [analysis $args]
                }
                "status" {
                    set result [status {*}$args]
                }
                "btconnect" {
                    set result [btconnect {*}$args]
                }
                "btdisconnect" {
                    set result [btdisconnect {*}$args]
                }
                "btremove" {
                    set result [btremove {*}$args]
                }
                "sql" {
                    set result [execSQL "$args"]
                }
                "recording" {
                    set result [recording {*}$args]
                }
                "session" {
                    set result [session {*}$args]
                }
                "btversion" {
                    set result [btversion {*}$args]
                }
                "version" {
                    # $args werden stillschweigend ignoriert.
                    set version [list wsevent "version" vmkstationd $::VERSION app $VERSION]
                    ::WSServer::disposeServerMessage apps/satteldruckanalyse text [::kvlist2json $version]
                }
                default {
                    set result [::apps::createJSONError client command "Unknown command: $commandname"]
                }
            }
        } error_msg]} {; # Fehler
            if {[info exists errorInfo]} {
                append error_msg "\n$errorInfo"
            }
            set result [::apps::createJSONError internal vmkstationd "$error_msg"]
        }
        if {"$result" != ""} {
            srvLog [namespace current] Warn "Client error: '$result'"
            ::WSServer::disposeServerMessage apps/satteldruckanalyse text $result
        } else {
            srvLog [namespace current] Info "Command executed: '$command'"
        }
        #}}}
    }; # proc handleCommand 


    ##{{{ init{}, start{}, stop{}

    # Initialisierung unmittelbar nach dem Laden
    proc init {} {; #{{{
        variable libdir
        variable APPQUESTIONS 

        srvLog [namespace current] Info "Initializing App ..."
        # "beckenrotation" wird nur im rotationsthread gebraucht
        #   und deshalb nur dort geladen.
        if {[catch {
            foreach libname {druckbildanalyse veloscore} {
                load $libdir/libtcl[info tclversion]${libname}.so
                srvLog [namespace current] Info "loaded: $libdir/libtcl[info tclversion]${libname}.so"
            }} msg]} {
                srvLog [namespace current]::init Error $msg
        }
        ::kernel::DB::initAppquestions satteldruckanalyse $APPQUESTIONS 
        srvLog [namespace current] Info "App initialized."
        #}}}
    }; # proc init 


    # Anwendung starten mit TTY
    proc start {} {; #{{{
        variable appsettings
        variable started

        srvLog [namespace current] Info "Starting App ..."
        # Callbacks setzen
        ::kernel::TTYSattel::addDriverHandler [namespace current]::handleUSBDriverChange
        ::kernel::BTSattel::addConnectionHandler [namespace current]::handleBTConnectionChange
        ::kernel::TTYSattel::addImageHandler [namespace current]::handleDBLDValues 
        # Der Imagehandler wird beim Eintreffen der ersten Daten auf BT umgeschaltet
        # und beim Verbindungsende zurück auf USB.
        # Evtl. vorhandene alte Bilddaten zurücksetzen
        resetData
        # Einstellungen holen
        set stored_settings [::kernel::DB::getAppsettings "satteldruckanalyse"]
        srvLog [namespace current]::init Debug "appsettings: $stored_settings"
        # Einstellungen übernehmen
        dict for {key value} $stored_settings {; # Gespeicherte Einstellungen durchgehen
            if {[llength $value] == 1} {
                srvLog [namespace current]::init Debug "Restore \"$key\": \"$value\""
                #TODO dict get kann fehlschlagen, wenn jemand an der Datenbank gepfuscht hat.
                #   => if {![dict exists ...]} {Warn continue}
                if {"[dict get $appsettings $key default]" != "$value"} {
                    srvLog [namespace current]::init Notice "Setting \"$key\" to \"$value\" ..."
                    switch $key {
                        "maskname" {
                            setMask $value
                        }
                        "compare" {
                            setCompare $value
                        }
                        "contrast" {
                            setJPEG colorcontrast $value
                        }
                        "bgcolor" {
                            setJPEG bgcolor $value
                        }
                        "create_jpeg" {
                            configSwitch create_jpeg $value
                        }
                        "create_norm" {
                            configSwitch create_norm $value
                        }
                        default {
                            srvLog [namespace current]::init Warn "Unknown setting \"$key\" not changed."
                        }
                    }
                }
                continue
            } else {; # Unterteilung nach display 0 ... 3
                dict for {display setting} $value {; # Displays durchgehen
                    srvLog [namespace current]::init Debug "Restore \"$key.$display\": \"$setting\""
                    #TODO dict get kann fehlschlagen, wenn jemand an der Datenbank gepfuscht hat.
                    #   => if {![dict exists ...]} {Warn continue}
                    if {"[dict get $appsettings $key $display default]" != "$setting"} {
                        srvLog [namespace current]::init Notice "Setting \"$key.$display\" to \"$setting\" ..."
                        switch $key {
                            "grid" {
                                setJPEG $display grid $setting
                            }
                            "showmask" {
                                if {$setting == "on"} {
                                    set setting "show"
                                } else {
                                    set setting "hide"
                                }
                                setJPEG $display mask $setting
                            }
                            "resolution" {
                                setJPEG $display resolution $setting
                            }
                            "frame" {
                                setJPEG $display frame $setting
                            }
                            "jpegquality" {
                                setJPEG $display quality $setting
                            }
                            default {
                                srvLog [namespace current]::init Warn "Unknown setting \"$key\" not changed."
                            }
                        }
                    }; # if Abweichung von default
                }; # dict for ... Displays durchgehen
            }; # else Unterteilung nach display 0 ... 3
        }; # dict for ... Gespeicherte Einstellungen durchgehen
        ::kernel::JPEGSattel::addImageHandler [namespace current]::jpegFinished {sda_live sda_int1 sda_int2 sda_int3}
        ::kernel::JPEGSattel::addOverloadHandler [namespace current]::handleOverload
        ::kernel::NormSattel::addNormHandler [namespace current]::normFinished {sda_live sda_int1 sda_int2 sda_int3}
        #TODO Overload ? (gibt es (noch?) nicht.)
        # Meldungen können noch nicht gesendet werden, weil die Websocketverbindung noch nicht steht.
        #  => künstlich verzögern
        after 1000 "
            # Maximum 0 schicken
            ::WSServer::disposeServerMessage apps/satteldruckanalyse text \"{\\\"wsevent\\\": \\\"max_value\\\", \\\"value\\\": 0}\"
            # Die aktuellen Bilder schicken
            [namespace current]::jpegRefresh 0 1 2
        "
        set started 1
        srvLog [namespace current] Info "App started."
        # (Die appquestions werden hier nicht (automatisch) gesendet, weil die Sprache nicht bekannt ist.)
        #}}}
    }; # proc start


    # Anwendung anhalten
    proc stop {} {; #{{{
        variable appsettings
        variable started
        variable recording
        variable rec_after_id

        srvLog [namespace current] Info "Stopping App ..."
        # Evtl. noch laufende Aufzeichnung abwürgen
        if {$recording} {
            after cancel "$rec_after_id"
            set rec_after_id ""
            set recording 0
            srvLog [namespace current]::stop Info "application stop while recording"
        }
        # Callbacks zurücknehmen
        ::kernel::TTYSattel::removeDriverHandler [namespace current]::handleUSBDriverChange
        ::kernel::BTSattel::removeConnectionHandler [namespace current]::handleBTConnectionChange
        #TODO Das gibt eine Warnung im Log, wenn der Callback nicht gesetzt war:
        ::kernel::TTYSattel::removeImageHandler [namespace current]::handleDBLDValues 
        ::kernel::BTSattel::removeDataHandler [namespace current]::handleDBLDValues 
        ::kernel::JPEGSattel::removeImageHandler [namespace current]::jpegFinished
        ::kernel::JPEGSattel::removeOverloadHandler [namespace current]::handleOverload
        ::kernel::NormSattel::removeNormHandler [namespace current]::normFinished
        #TODO Overload entfern, falls es das 'mal geben sollte
        set started 0
        # appsettings ohne defaults speichern
        set as2store $appsettings
        dict for {outer value} $appsettings {
            if {[dict exists $value "default"]} {
                dict unset as2store $outer "default"
            } else {
                dict for {inner dummy} $value {
                    dict unset as2store $outer $inner "default"
                }
            }
        }
        srvLog [namespace current]::stop Debug "settings to store:\n$as2store"
        ::kernel::DB::setAppsettings "satteldruckanalyse" $as2store
        srvLog [namespace current] Info "App stopped."
        #}}}
    }; # proc stop 

    ##}}} init{}, start{}, stop{}
    
}; # namespace eval satteldruckanalyse

set app_loaded satteldruckanalyse

