#!/usr/bin/tclsh
#
# Velometrik Arbeitsstationsdeamon
#
# Ermöglicht die Steuerung einer Arbeitsstation mittels Browser.
#
# Author: Siegmar Müller
#
# Historie:
# ??.06.2019 Siegmar Müller Testversion fertiggestellt
# 05.05.2022 Siegmar Müller vmkstationd3 begonnen
#


set ::dir [file dirname [info script]]
if {[string equal -length 1 $::dir "."]} {
    set dir "$env(PWD)[string range $::dir 1 end]"
}

#set VERSION 3.0.0_A
set VERSION "3.0.0_B2"
set CONF_PORT 8080
set CONF_LOGFILE "$::dir/vmkstationd.log"
set CONF_LOGLEVEL debug; # TODO notice oder so falls nicht Alpha
set JPEGOPTIONS [list -quality 60 -res 12 -grid no]
set PIEPSER 0; # Wird nach erfolgreicher Initialisierung 1
# Folgendes wird von der Initialisierung überschrieben:
set BGCOLOR #404040
set WS_URL "https://ws-1-1.velometrik.de/vmkservice"
set STATION_NR 3; # Teststation => Das muß überschrieben werden!
set MIN_VALUE_HOCKER 0; # Mindestwert vom Hocker-Controller unterhalb dem alles 0 gesetzt wird

# Workaround für ein Problem, das mit tcllib1.19 aufgetreten ist:
if {![info exists ::env(USER)]} {
    set ::env(USER) root
}
if {![info exists ::env(user)]} {
    set ::env(user) $::env(USER)
}
#Ende: Workaround


# Globale Daten
set ::SODIR "." ; # Verzeichnis der shared objects (wird noch ermittelt)
set ::BINDIR "/usr/local/bin" ; # Verzeichnis der ausführbaren Dateien (wird noch ermittelt)
set ::PORTALSERVER "" ; # Wird vom Webservice geliefert
# (siehe auch setup/vmkstationd.conf)
array set ::LEDFARBEN [list SCHWARZ 0 GRUEN 1 BLAU 2 HELLBLAU 3 ROT 4 GELBGRUEN 5 VIOLETT 6 WEISS 7]
# LAN beinhaltet WLAN
set ::STATUSLEVELS [list OFF POWER MATTE LAN INET WEBSERVICE ANGEMELDET]
array set ::STATUSFARBEN [list OFF SCHWARZ POWER ROT MATTE VIOLETT LAN BLAU INET HELLBLAU WEBSERVICE GRUEN ANGEMELDET WEISS]
# Der aktuelle Stationsstatus (OFF darf niemals 0 sein)
array set ::stationsstatus [list OFF 1 POWER 0 MATTE 0 LAN 0 INET 0 WEBSERVICE 0 ANGEMELDET 0]

#{{{ Konfiguration holen
if {[file exists /usr/local/etc/vmkstationd.conf]} {
        source /usr/local/etc/vmkstationd.conf
        if {[info exists PORT]} {
            set CONF_PORT $PORT
        }
        if {[info exists LOGLEVEL]} {
            set CONF_LOGLEVEL $LOGLEVEL
        }
        # Noch nicht möglich: srvLog {} Notice "$::dir/start/vmkstationd.conf loaded"
} else {
    puts stderr "/usr/local/etc/vmkstationd.conf doesn't exist."
}

#TODO Kodemüll (Kann 'raus)
if {[file isdirectory $::dir/start]} {
} else {
    puts stderr "Verzeichnis $::dir/start existiert nicht."
}
#}}} Startkode holen und ausführen

# Hilfe ausgeben und beenden
proc help {} {; #{{{
    puts ""
    puts "Velometrik Arbeitsstationsdeamon V $::VERSION"
    puts "Startparameter:"
    puts "    -p <port>     Start auf Port <port> (Vorgabe $::CONF_PORT)"
    puts "    -L <logfile>  Logfile (Vorgabe: $::CONF_LOGFILE)"
    puts "                  ('' => Ausgabe an stderr)"
    puts "    -l <loglevel> Loglevel (error warn notice info debug)"
    puts "                  (Vorgabe: $::CONF_LOGLEVEL)"
    puts "    -V        Version ausgeben und beenden"
    puts "    -h        Hilfe ausgeben und beenden"
    puts "    -?        Hilfe ausgeben und beenden"
    puts ""
    exit
    #}}}
}; # proc help


#{{{ Kommandozeilenparameter übernehmen
set status option
foreach arg $argv {
    switch -- $status {
        option {
            switch -- $arg {
                -p {
                    set status "-p"
                }
                -L {
                    set status "-L"
                }
                -l {
                    set status "-l"
                }
                -V {
                    puts $VERSION
                    exit
                }
                -h {
                    help
                    exit
                }
                -? {
                    help
                    exit
                }
                default {
                    puts stderr "Unbekannte Option: $arg"
                    exit 1
                }
            }; # switch $arg
        }
        "-p" {
            #TODO Integer?
            #TODO Rootrechte erforderlich?
            set CONF_PORT $arg
            set status option
        }
        "-L" {
            set CONF_LOGFILE $arg
            set status option
        }
        "-l" {
            if {[lsearch -exact  [list "error" "warn" "notice" "info" "debug"] $arg] < 0} {
                puts stderr "Unbekanntes Loglevel: '$arg'"
                exit 1
            }
            set CONF_LOGLEVEL $arg
            set status option
        }
    }; # switch $status
}
if {"$status" != "option"} {
    puts stderr "Wert für Option $status fehlt."
    exit 1
}
#}}} Kommandozeilenparameter sind übernommen


#TODO Die global benötigten package require sollten an geeigneter Stelle zusammengefaßt werden.
package require http
package require json
package require Thread

# Letztes Logfile, falls vorhanden sichern.
proc logRestart {} {
    if {[file exists $::CONF_LOGFILE]} {
        set fd [open $::CONF_LOGFILE "a"]
        puts $fd "[clock format [clock seconds]] logRestart: ${::CONF_LOGFILE}~ wird erstellt"
        close $fd
        file copy -force $::CONF_LOGFILE ${::CONF_LOGFILE}~
        file delete $::CONF_LOGFILE
    }
    if {[llength [info procs srvLog]] > 0} {; # Kein Neustart, weil utils.tcl bereits geladen
        srvLog {} Notice "logRestart"
    }
    # Das nächste mal um Mitternacht
    after [expr {([clock add [clock scan 23:59] 1 minutes] - [clock seconds]) * 1000}] logRestart
}

logRestart


#{{{ Startkode holen und ausführen
if {[file isdirectory $::dir/start]} {
    foreach file [glob $::dir/start/*.tcl] {
        source $file
        #TODO Das geht schief, wenn utils.tcl nicht zuerst geholt wird.
    # => Dateien numerieren, wenn es mehrere werden.
        srvLog {} Notice "$file loaded"
    }
} else {
    puts stderr "Verzeichnis $::dir/start existiert nicht."
}
#}}} Startkode holen und ausführen

setLoglevel $CONF_LOGLEVEL

#{{{ TCL-Bibliotheken holen
set machine [exec uname -m]
if {[file isdirectory $::dir/bin/$machine]} {
    set ::BINDIR $::dir/bin/$machine
}
if {[file isdirectory $::dir/lib]} {
    if {[ file isdirectory $::dir/lib/$machine]} {
        #   shared Objects
        set ::SODIR $::dir/lib/$machine
        foreach file [glob -nocomplain $::SODIR/libtcl${tcl_version}*.so] {
            load $file
            srvLog {} Notice "$file loaded"
        }
    } else {
        srvLog {} Warn "$::dir/lib/${machine}/ doesn't exist."
    }
    #   reine TCL-Bibliotheken
    foreach file [glob -nocomplain $::dir/lib/*.tcl] {
        source $file
        srvLog {} Notice "$file loaded"
    }
} else {
   srvLog {} Warn "$dir/lib/ doesn't exist."
}
#}}} TCL-Bibliotheken holen

# Bibliotheken konfigurieren
if {[namespace exists ::TTYHocker]} {
    set ::TTYHocker::MIN_VALUE $::MIN_VALUE_HOCKER
    srvLog {} Notice "MIN_VALUE_HOCKER = $::MIN_VALUE_HOCKER"
}

# Workaround zum Unterbinden der unnötig zeitverzögernden Hostabfrage bei nicht erreichbarem DNS-Server
proc ::websocket::fconfigure {args} {
    if {[lindex $args 1] in {-peername -sockname}} {
        upvar #0 clientcontext[lindex $args 0] context
        set ipaddr $context(ipaddr)
        set port $context(port)
        srvLog {Workaround} Debug "fconfigure $args => ${ipaddr}:$port"
        return [list $ipaddr $ipaddr $port]
    }
    return [::fconfigure {*}$args]
}; # proc ::websocket::fconfigure

# Behandlung von Hintergrundfehlern
# Wird aufgerufen, wenn der Interpreter (bei vwait) in einen vorher nicht aufgefangenen Fehler läuft:
# @param msg    Meldung vom TCL-Kern
proc bgerror {msg} {
    global errorInfo

    srvLog bgerror Error "$msg\n$errorInfo"
}; # proc bgerror

# Fehlerbehandlung für Threads
proc threaderror {thread_id errorInfo} {
    srvLog [namespace current] Error "Thread id=$thread_id: $errorInfo"
}
thread::errorproc [namespace current]::threaderror 

##################################################################################################
proc tracecommand {command op} {
    srvLog trace Debug "$op: $command"
}
# trace add execution ::websocket::upgrade enterstep tracecommand
##################################################################################################

# Status setzen/zurücksetzen
#TODO Unklar: Welche Bedeut hat die Sattelmatte für den Stationsstatus?
#TODO Dok fertig
proc setStationsstatus {statusname {on_off 1}} {; #{{{
    set i_status [lsearch $::STATUSLEVELS $statusname]
    if {$i_status < 0} {; # interner Fehler
        return; # wird ignoriert.
    }
    set ::stationsstatus($statusname) $on_off
    set i_status 0
    foreach statuslevel $::STATUSLEVELS {
        if {!$::stationsstatus($statuslevel)} {
            break
        }
        incr i_status
    }
    incr i_status -1
    set status_neu [lindex $::STATUSLEVELS $i_status]
    if {$::PIEPSER} {
        ::piepser::setLED $::LEDFARBEN($::STATUSFARBEN($status_neu))
    }
    #TODO Statusänderung über Websocket verteilen
    srvLog {setStationsstatus} Notice "Stationsstatus: $status_neu"
    #}}}
}; # proc setStatus 


# Piepser mit LEDs initialisieren
# Das scheitert, wenn es nicht auf einem Raspi ausgeführt wird.
if {[catch {::piepser::gpioSetup} msg]} {
    srvLog gpioSetup Error "$msg"
} else {
    set PIEPSER 1
    srvLog gpioSetup Notice "Piepser initialisiert"
    setStationsstatus POWER
}


# Beendete Messung signalisieren
proc signalFinished {} {
    if {$::PIEPSER} {
        # Ein Fehler an dieser Stelle ist unproblematisch,
        # darf aber die Weitergabe der Meßergebnisse nicht blockieren.
        catch {::piepser::toogleGpioValue $::piepser::BUZZER {100 100 800 2000}}
    }
}; # proc signalFinished 


#TODO Konfigurierbar:
set ::WSServer::HTTPD(default) "index.tcls"


# Prüft die Apache Proxykonfiguration für den eingestellten Portalserver (Variable ::PORTALSERVER),
# ändert sie bei Bedarf und startet Apache neu.
# Verwendete Konfiguration: /etc/apache2/sites-available/000-default.conf
proc checkApacheConf {} {; #{{{
    if {![file exists /etc/apache2/sites-available/000-default.conf]} {
        srvLog checkApacheConf Warn "/etc/apache2/sites-available/000-default.conf existiert nicht."
        return
    }
    if {[catch {
        # Portalserver der aktuellen Konfiguration extrahieren
        set aktuell [exec sed -En {/ProxyPass "\/stationsproxy"/s/^.*"(http.*)"/\1/p} /etc/apache2/sites-available/000-default.conf]
        if {"$aktuell" == "$::PORTALSERVER"} {
            srvLog {} Info "Prüfung der Apache Proxykonfiguration: /stationsproxy = $aktuell = Portalserver (OK) "
        } else {
            srvLog {} Info "Prüfung der Apache Proxykonfiguration: /stationsproxy = $aktuell != Portalserver ($::PORTALSERVER) "
            file copy -force /etc/apache2/sites-available/000-default.conf /etc/apache2/sites-available/000-default.conf~
            # Konfiguration anpassen:
            exec sed -E "/ProxyPass(Reverse)? \"\\/stationsproxy\"/s/\"http.*\"/\"[string map {/ \\/} $::PORTALSERVER]\"/" /etc/apache2/sites-available/000-default.conf~ > /etc/apache2/sites-available/000-default.conf
	    #TODO CookieDomain muß ebenso und zusätzlich mit local_ip angepaßt werden.
            srvLog {} Notice "$::PORTALSERVER in Apache Proxykonfiguration eingetragen"
            # Apache Neustart endet mit Fehler, wenn kein Servername bestimmt werden kann.
            # Deshalb in /etc/apache2/apache2.conf eintragen:
            # ServerName intranet.velobox.com
            exec apachectl graceful
            srvLog {} Notice "Apache neu gestartet"
        }
    } error]} {
        srvLog {} Error $error
    }
    #}}}
}; # proc checkApacheConf 


# Kommunikation mit dem Webservice
# Prüft die Internetverbindung und meldet sich beim Webservice
namespace eval ::wscomm {; #{{{
    variable local_ip ""
    variable netzwerk_aktiv 0
    variable internet_verfuegbar 0
    variable webservice_verfuegbar 0
    variable after_id ""; # für die Wiederholungsversuche

    # Lokales Netzwerkinterface testen.
    # varmsg    Name der Variablen zum Hinterlegen der Meldung
    # => 1  Das Interface koennte funktionieren.
    #    0  Es wurde ein Fehler festgestellt.
    proc localIfcheck {msgvar} {; #{{{
        upvar $msgvar msg
        variable local_ip
    
        set driver ""
        if {[catch {
                foreach line [split [exec ip -4 address] "\n"] {
                    if {[regsub {^ *inet ([0-9.]*)/.*$} $line {\1} temp]} {; # temp hat local_ip
                        set local_ip $temp
                        if {"$driver" == "lo"} {
                            continue
                        } elseif {[string match "eth*" $driver]} {; # ... hat Vorrang
                            break
                        }
                    }
                    if {[regsub {^.* ([a-z][a-z0-9]*): .*$} $line {\1} temp]} {; # temp hat driver
                        set driver $temp
                        continue
                    }
                }
            } errmsg]} {
                set msg "LOCAL_IP kann nicht ermittelt werden: $errmsg"
                return 0
        } else {
            if {"$driver" == "lo"} {
                set msg "Keine Netzwerkverbindung"
                setStationsstatus LAN 0
                return 0
            } else {
                set msg "$driver: LOCAL_IP=$local_ip"
                setStationsstatus LAN
                return 1
            }
        }
        #}}}
    }; # proc localIfcheck


    # HTTP-Request ausfuehren
    # httpUrl   Vollstaendiger Request
    # datavar   Variable zum Hinterlegen der Antwort bzw. der Fehlermeldung
    # =>    1 Request wurde ausgefuehrt, datavar enthaelt das Ergebnis
    #       0 datavar enthaelt eine Fehlermeldung oder ist leer, wenn die Fehlermeldung zuvor schon uebergeben wurde
    proc execHttpRequest {httpUrl datavar} {; #{{{
        upvar $datavar data
        variable last_ws_error
        variable last_user_error

        set ok 0
        if {[catch {
            set tokenHttp [::http::geturl $httpUrl]; # Wirft evtl. eine Exception
            # Verbindung konnte hergestellt werden, falls hier angekommen.
            ::http::wait $tokenHttp
            set err_msg ""
            # http-Antwort auswerten
            switch [::http::status $tokenHttp] {
                ok {
                    if {[::http::ncode $tokenHttp] == 200} {
                        set data [::http::data $tokenHttp]
                        set ok 1
                    } else {
                        set err_msg "#WS: [::http::code $tokenHttp]"
                    }
                }
                eof {
                    set err_msg "#EOF: Connection closed by Webservice."
                }
                error {
                    set err_msg "#ERROR: [::http::error $tokenHttp]"
                }
            }
            ::http::cleanup $tokenHttp

            if {!$ok} {
                # Exception erst hier werfen, damit ::http::cleanup noch ausgefuehrt wird.
                error $err_msg
            }
        } catchResult]} {
            srvLog {} Error $catchResult
        }
        return $ok
        #}}}
    }; # execHttpRequest


    # Abarbeitung eines Webservice Requests
    # request   Bezeichnung des Requests
    # lquery    Parameter als Liste [list parameter wert ...]
    # datavar   Variable fuer das Ergebnis
    # =>    1 Request wurde ausgefuehrt, datavar enthaelt das Ergebnis
    #       0 datavar enthaelt eine Fehlermeldung oder ist leer, wenn die Fehlermeldung zuvor schon uebergeben wurde
    proc execWsRequest {request lquery datavar} {; #{{{
        upvar $datavar data

        set parameter "[::http::formatQuery {*}[join $lquery " "]]"
        if {[catch {
            srvLog {} Debug "Encrypting $parameter"
            set encrypted [vmkcrypt encrypt $parameter]
        } errmsg ]} {
            set data ""
            srvLog {} Error $errmsg
            return 0  
        }
        # Da keine Sonderzeichen mehr vorkommen koennen, kann das direkt zusammengebaut werden.
        #TODO Request kann nicht ausgeführt werden,
        # wenn STATION_NR == 0, d.h. nicht initialisiert ist.
        set parameter "sn=$::STATION_NR&req=$encrypted"
        srvLog {} Debug "execWsRequest: ${::WS_URL}/${request}?$parameter"
        return [execHttpRequest "${::WS_URL}/${request}?$parameter" data]
        #}}}
    }; # proc execWsRequest 


    # Ruft localIfcheck auf und prüft bei Erfolg den Internetzugang durch ping auf Google.
    # Wenn das erfolgreich war, wird dem Webserver die ermittelte local_ip mitgeteilt
    # und die zurückgegebene URL in ::PORTALSERVER gespeichert.
    # Falls diese Kette nicht fehlerfrei durchlaufen wurde, enthält ::PORTALSERVER einen Leerstring.
    # @param retry Anzahl der Wiederholungsversuche nach (10s) wenn Schnittstelle aktiv, aber keine Internetverbindung
    proc inetcheck {{retry 5}} {; #{{{
        variable local_ip
        variable netzwerk_aktiv
        variable internet_verfuegbar
        variable webservice_verfuegbar
        variable after_id

        set ::PORTALSERVER ""
        if {$after_id != ""} {
            after cancel $after_id
        set after_id ""
        }
        if {[localIfcheck nwmsg]} {
            # Lokale Schnittstelle ist aktiv.
            set netzwerk_aktiv 1
            srvLog {inetcheck} Notice "Lokale Netzwerkschnittstelle ist aktiv ($nwmsg)"
            # => Verfuegbarkeit des Internets pruefen
            #    Das geschieht mittels ping auf den Google DNS-Server 8.8.8.8.
            if {![catch {exec ping -c1 8.8.8.8} result]} {
                srvLog {inetcheck} Notice "Internetzugang O.K."
                set internet_verfuegbar 1
                setStationsstatus INET
                # Antwortet der Webservice ?
                # (Bei der Gelegenheit setzen wir gleich die IPadresse.)
                if {[execWsRequest set [list station_nr $::STATION_NR what local_ip value $local_ip] wsAntwort]} {
                    srvLog {inetcheck} Debug "Webservice antwortet auf /set local_ip mit \"$wsAntwort\"."
                    if {[regexp {^{.*}$} $wsAntwort]} {; # JSON-Antwort
                        set dictWsAntwort [::json::json2dict $wsAntwort]
                        if {[dict exists $dictWsAntwort portalserver]} {
                            set ::PORTALSERVER [dict get $dictWsAntwort portalserver]
                            srvLog {inetcheck} Notice "Portalserver ist $::PORTALSERVER."
                            setStationsstatus ANGEMELDET
                            #TODO checkApacheConf
                        } else {
                            srvLog {inetcheck} Warn "Webservice meldet keinen Portalserver."
                            setStationsstatus ANGEMELDET 0
                        }
                    }
                    set webservice_verfuegbar 1
                    setStationsstatus WEBSERVICE
                } else {
                    srvLog {inetcheck} Warn "Webservice nicht verfügbar"
                    set webservice_verfuegbar 0
                    setStationsstatus WEBSERVICE 0
                }
            } else {; # Keine Antwort auf ping
                srvLog {inetcheck} Warn "Kein Internetzugang"
                set internet_verfuegbar 0
                setStationsstatus INET 0
            }
        } else {; # Fehler
            srvLog {inetcheck} Error $nwmsg
        set netzwerk_aktiv 0
        }
        if {$netzwerk_aktiv && !$internet_verfuegbar && $retry} {
            # Weiter probieren
            set after_id [after 30000 ::wscomm::inetcheck [expr $retry - 1]]
        } else {
            srvLog {inetcheck} Notice "Keine weiteren Verbindungsversuche zum Internet"
        }
        #}}}
    }; # proc inetcheck

    # Verfuegbarkeit des Netzwerkes und des Webservice pruefen
    # Vorsichtshalber etwas warten, falls der Bootvorgang noch nicht so weit ist.
    after 5000 ::wscomm::inetcheck

    #}}}
}; # namespace eval ::wscomm


### FIFO für lokalen Clientzugriff
# Benutzt zum Reagieren auf Änderungen im Netzwerk
# Nur zu experimentellen Zwecken: Loglevel ändern, Deamon abwürgen
#{{{

# Eine über den FIFO eingegangene Meldung baerbeiten
proc handleFifoRequest {} {
    set request [gets $::fd_fifo]
    srvLog FIFO Info "Request eingegangen: $request"
    if {[string equal -length 6 $request "dhcpcd"]} {
        # dhcpcd start/stop
        after 1500 ::wscomm::inetcheck
    } elseif {[string equal -length 5 "$request" "loglevel"]} {
        setLoglevel [regsub {loglevel *} $request {}]
    } elseif {"$request" == "terminate"} {
        srvLog FIFO Notice "\"terminate\" empfangen."
        exit
    }
}; # proc handleFifoRequest 

if {[catch {
        exec /bin/mkdir -p -m 775 /var/local/run/vmkstationd
        if {![file exists /var/local/run/vmkstationd/vmkstationd.fifo]} {
            exec /usr/bin/mkfifo /var/local/run/vmkstationd/vmkstationd.fifo
        }
        set ::fd_fifo [open /var/local/run/vmkstationd/vmkstationd.fifo r+]
        fconfigure $::fd_fifo -blocking 0
        fileevent $::fd_fifo readable handleFifoRequest 
    } result]} {
    srvLog {} Error "FIFO-Kommunikation ist nicht möglich: $result"
} else {
    srvLog {} Notice "Lausche auf /var/local/run/vmkstationd/vmkstationd.fifo"
}

###}}} FIFO für lokalen Clientzugriff


# Globale Daten der Station
#TODO Das kommt nach TTYSattel und TTYHocker
array set ::matten {}


# Callback für die Treiberüberwachung
# @param change Die Änderung als dict mit folgenden Schlüsseln:
#TODO Das verschwindet in TTYSattel::driverChanged (erledigt) und TTYHocker::driverChanged (offen)
proc driverChanged {change} {; #{{{
    set driver [dict get $change "driver"]
    switch [dict get $change "action"] {
        + {
            set type [dict get $change "type"]
            switch -glob $type {
                MT_STD {
                    if {[::TTYHocker::start $driver [dict get $change "speed"] ::hocker::nextImage]} {
                        set ::matten($driver) $type
                        setStationsstatus MATTE 
                    }
                }
                MT_SAT* {
                    if {[::TTYSattel::start $driver [dict get $change "speed"] ::sattel::nextImage]} {
                        set ::matten($driver) $type
                    }
                }
            }
            # Meldung an die Clients
            set msg [list wsevent plugged driver $driver type $type]
        }
        - {
            if {![info exists ::matten($driver)]} {
                srvLog {} Notice "Zuvor nicht registrierter Treiber $driver entfernt."
                return
            }
            # Meldung an die Clients
            set type $::matten($driver)
            set msg [list wsevent unplugged driver $driver type $type]
            unset ::matten($driver)
            if {$type == "MT_STD"} {
                setStationsstatus MATTE 0
            }
        }
        default {
            return
        }
    }; # switch action
    # Meldung an die WS-Clients als JSON-Objekt abschicken.
    ::WSServer::disposeServerMessage messages text [::kvlist2json $msg]
    #}}}
}; # proc driverChanged 

namespace eval ::kernel {
    srvLog {::kernel} Notice "::kernel wird initialisiert"
    ### Kernelmodule für alle Anwendungen
    # (Die anwendungsspezifischen Module können sich auf das Vorhandensein dieser Module verlassen.)
    if {[catch {
        srvLog {::kernel} Notice "Lade lib/kernel/watchttyacm.tcl"
        source $::dir/lib/kernel/watchttyacm.tcl; # namespace watchTTYACM
        srvLog {::kernel} Notice "lib/kernel/watchttyacm.tcl geladen"
    } msg]} {
        srvLog {::kernel} Error $msg
    }

    ### Anwendungsspezifische Kernelmodule
    # (Die Anwendungen können sich auf das Vorhandensein dieser Module verlassen.)
    # (Kernel-)Module laden
    set mods_loaded [list]; # Die geladenen Module müssen anschließend initialisiert werden.
    set mod_loaded ""; # Das muß das Modul beim Laden setzen.
    foreach tclfile [glob $::dir/custom/kernel/*.tcl] {
        
        if {[catch {source $tclfile} load_error]} {
            srvLog {::kernel} Error "$load_error\n${::errorInfo}"
            continue
        }
        if {"$mod_loaded" == ""} {
            srvLog {::kernel} Error "mod_loaded nicht gesetzt von $tclfile"
        } else {
            lappend mods_loaded $mod_loaded
            srvLog {::kernel} Notice "$mod_loaded geladen aus $tclfile"
        }
    }
    # Module initialisieren
    foreach mod_loaded $mods_loaded {
        if {"[info procs ${mod_loaded}::init]" != ""} {
            if {[catch {${mod_loaded}::init} init_error]} {
                srvLog {::kernel} Error $init_error
            } else {
                srvLog {::kernel} Info "$mod_loaded initialisiert"
            }
        } else {
            srvLog {::kernel} Warn "${mod_loaded}::init existiert nicht. Modul wird nicht initialisiert."
        }
    }

    srvLog {::kernel} Debug "Kernelmodule: [namespace children]"

    # Server und Kernelmodule starten
    proc start {} {; # (neu)
        # HTTP/WS-Server starten
        srvLog {} Notice "Starte Arbeitsstationsdämon auf Port $::CONF_PORT ..."
        ::WSServer::start "port $::CONF_PORT doc_root $::dir/htdocs"
        srvLog {} Notice "... gestartet."
        # Treiberüberwachung starten
        watchTTYACM::start
    }

}; # ::kernel

::kernel::start
srvLog {::kernel} Notice "::kernel gestartet"


#TODO Das wird zerlegt in das Kernelmodul JPEGHocker in custom/kernel/jpeghocker.tcl und die App Sitzknochenabstand in custom/apps/sitzknochenabstand.tcl
namespace eval ::hocker {; #{{{
    variable N_AVG 5; # Anzahl der Bilder, über die der Durschnitt gebildet wird
    variable druckbilder [list]
    variable vsums [list]
    variable n_druckbilder 0
    variable i_druckbild 0
    variable jpeg_image [druckbild::jpegdata {*}$::JPEGOPTIONS [lrepeat [expr 16*28] 0] 16 28]
    variable dbld_analyse [list]; # für httpdomains::storedbld
    variable analyse [dict create]
    variable analyse_laeuft 0

    # Callback für die Weitergabe der Druckwerte von ::TTYHocker
    #TODO Parameter
    # @param v_sum  Summe der Druckwerte
    proc nextImage {druckbild n_rows n_cols v_sum} {; #{{{
        variable N_AVG
        variable druckbilder
        variable vsums
        variable n_druckbilder
        variable i_druckbild
        variable jpeg_image
        variable dbld_analyse
        variable analyse
        variable analyse_laeuft

        set start_analyse 0
        if {$v_sum > 0} {; # Prüfe, ob Auswertung sinnvoll ist
            # Bild festhalten
            lset druckbilder $i_druckbild $druckbild
            # v_sum festhalten
            lset vsums $i_druckbild $v_sum
            # Prüfen, ob eine Auswertung sinnvoll ist
            # Auswerten ist sinnvoll bei genug Bildern und wenig Gezappel.
            # Statt zeitaufwendig den Durchschnitt der Bilder zu berechnen, nehmen wir das Druckbild,
            # dessen Drucksumme am nächsten beim Durchschnitt liegt. ($i_min_gezappel)
            if {$n_druckbilder >= $N_AVG} {
                set v_avg 0
                foreach vsum $vsums {
                    incr v_avg $vsum
                }
                set v_avg [expr $v_avg / $N_AVG]
                set start_analyse 1
                set min_gezappel 10
                set i_min_gezappel 0
                set i_vsum 0
                foreach vsum $vsums {
                    set gezappel [expr abs($vsum - $v_avg) / double($v_avg)]
                    if {$gezappel > 0.1} {
                        # Noch zu viel Gezappel.
                        set start_analyse 0
                        break
                    }
                    if {$gezappel < $min_gezappel} {
                        set min_gezappel $gezappel
                        set i_min_gezappel $i_vsum
                    }
                    incr i_vsum
                }
            }
            # Weiter mit nächstem Druckbild
            set i_druckbild [expr ($i_druckbild + 1) % $N_AVG]
            if {$n_druckbilder < $N_AVG} {
                incr n_druckbilder 
            }
        } else {; # Kein Druck => von vorn anfangen
            set n_druckbilder 0
            set i_druckbild 0
            set analyse_laeuft 0
        }
        if {$start_analyse} {; # Auswertung ist sinnvoll.
            # Analyse ausführen und Ergebnis festhalten
            set druckbild [lindex $druckbilder $i_min_gezappel]
            set analyse [druckbild::analyse std $druckbild $n_rows $n_cols]
            set analyseerfolg [expr [dict get $analyse schwerpunkt1_x]>0 && [dict get $analyse schwerpunkt1_y]>0 && [dict get $analyse schwerpunkt2_x]>0 && [dict get $analyse schwerpunkt2_y]>0]
            if {$analyseerfolg} {; # als Sitzknochenabstand weitergeben
                dict set analyse wsevent sitzknochenabstand
            } else {; # nur als Hockerbild weitergeben
                dict set analyse wsevent hockerbild
            }
            dict set analyse v_sum $v_sum
            if {!$analyse_laeuft} { 
                if {$analyseerfolg} {
                    # Erstes Analyseergebnis der aktuellen Messung
                    # => Signal erzeugen
                    signalFinished 
                    set analyse_laeuft 1
                }
            } else {; # Analyse läuft bereits
                if {!$analyseerfolg} {; # Die Analyse war aber nicht mehr möglich.
                    set analyse_laeuft 0
                }
            }
            # Druckbild für Abruf (Mailversand) festhalten
            if {[namespace exists ::WSServer::httpdomains::storedbld]} {
                set ::WSServer::httpdomains::storedbld::hockerbild $druckbild
                #srvLog {} Debug [join $druckbild " "]
            } else {
                srvLog {} Debug "::WSServer::httpdomains::storedbld doesn't exist."
            }
        } elseif {!$analyse_laeuft} {; # Bild ohne Analyse bereitstellen, falls noch keine Analyse begonnen hat
            set analyse [dict create wsevent hockerbild v_sum $v_sum]
        } else {; # Bei austrudelnder Analyse nichts ausgeben
            return
        }
        srvLog {} Debug "::hocker::nextImage BxH=$n_cols*$n_rows $analyse"
        # JPEG-Generierung starten
        ::DBLD2IMG::startJPEG ::hocker::jpegFinished $druckbild $n_rows $n_cols
        #}}}
    }; # proc nextImage


    # Callback für die Weitergabe des erzeugten JPEG-Bildes
    proc jpegFinished {} {
        variable jpeg_image
        variable analyse_laeuft
        variable analyse

        set jpeg_image [::tsv::get images jpegimage]
        srvLog {} Debug "=> JPEG [string length $jpeg_image] Bytes"
        #TODO nur wenn das eingeschaltet ist:
        ::WSServer::disposeServerMessage messages text [::kvlist2json [dict get $analyse]]
    }

    #}}}
}; # namespace eval ::hocker


# Sattelbildanzeige (TODO feste App?)
# Das ist jetzt das Kernelmodul JPEGSattel in custom/kernel/jpegsattel.tcl
#namespace eval ::sattel {; #{{{
#    variable jpeg_image [druckbild::jpegdata {*}$::JPEGOPTIONS [lrepeat [expr 28*16] 0] 28 16]
#    variable dbg2counter 0
#
#    # Callback für die Weitergabe der Druckwerte von TTYSattel
#    proc nextImage {druckbild n_rows n_cols} {
#        variable dbg2counter
#
#        if {$dbg2counter % 30 == 0} {
#            srvLog {} Debug "::sattel::nextImage BxH=$n_cols*$n_rows"
#        }
#        # incr dbg2counter in jpegFinished
#        ::DBLD2IMG::startJPEG [namespace current]::jpegFinished $druckbild 28 16
#    }
#    
#    
#    # Callback für die Weitergabe des erzeugten Druckbildes vom Hocker
#    proc jpegFinished {} {
#        variable jpeg_image
#        variable dbg2counter
#
#        set jpeg_image [::tsv::get images jpegimage]
#        # Die Sattelmatte liefert, auch wenn sie nur herumliegt, 12 Bilder je Sekunde.
#        # Das wird wird selbst im Loglevel "Debug" zu viel.
#        if {$dbg2counter % 30 == 0} {
#            srvLog {jpegFinished} Debug "=> JPEG [string length $jpeg_image] Bytes"
#        }
#        incr dbg2counter
#        # nur wenn das eingeschaltet ist:
#        ::WSServer::disposeServerMessage messages text "{\"wsevent\": \"sattelbild\"}"
#    }
#    #}}} namespace eval ::sattel
#}

### Die Anwendungen
#TODO Doc zu den Konventionen (Variable, Prozeduren)
namespace eval ::apps {
    variable apps_loaded [dict create]; # Schlüssel App, Wert Filename

    srvLog "[namespace current]" Notice "::apps werden initialisiert"

    # Fehlermeldung als wsevent in JSON erstellen
    # Die erstellten Schlüssel sind:
    #   "wsevent" mit Wert "error"
    #   "class" Fehlerklasse
    #   "source"  Fehlerquelle
    #   "nr"    Fehlernummer
    #   "msg"   Eine Fehlermeldung
    # Derzeit definierte Fehlernummern 
    #   1 Insufficient data (for analysis)
    #   2 Still recording
    #   3 Unstored recording(s)
    #   4 No session
    #   5 No finished recording
    # @param class_nr   Liste aus Fehlerklasse (internal|client|user) und nr
    # @param msg        Text
    proc createJSONError {class_nr source msg} {; #{{{
        srvLog [namespace current] Debug "createJSONError '$class_nr' '$msg'"
        set class [lindex $class_nr 0]
        set nr [lindex $class_nr 1]
        # Gültige Fehlerklasse?
        if {"$class" ni {internal client user}} {
            srvLog [namespace current] Debug "createJSONError Unknown error class: \"$class\""
        }
        set msg [string map {\" \\\"} $msg]
        set kvlist [list "wsevent" "error" "class" $class "source" "$source" "msg" $msg]
        if {"$nr" != ""} {
            lappend kvlist nr $nr
        }
        set json_error [::kvlist2json $kvlist]
        return $json_error
        #}}}
    }; # proc createJSONError 

    # App neu laden
    # @return Fertigmeldung
    proc reload {app} {; #{{{
        variable apps_loaded
 
        if {[catch {
            # Applikation stoppen, falls gerade am Laufen
            if {[info exists ::apps::${app}::started]} {
                set started [set ::apps::${app}::started]
            } else {
                set started 0
            }
            if {$started} {
                if {"[info procs ::apps::${app}::stop]" != ""} {
                    ::apps::${app}::stop
                    append msg "$app stopped\n"
                }
            }
            set tclfile [dict get $apps_loaded $app]
            set app_loaded ""
            source $tclfile
            if {"$app_loaded" == ""} {
                append msg "Error: app_loaded not set by $tclfile"
            } else {
                append msg "$app_loaded loaded from $tclfile"
                # Initialisieren
                if {"[info procs ${app_loaded}::init]" != ""} {
                    ${app_loaded}::init
                    append msg "\n$app_loaded initialisiert"
                } else {
                    append msg "\n${app_loaded}::init existiert nicht. App wird nicht initialisiert."
                }
                # neu starten, falls vorher auch gestartet
                if {$started} {
                    if {"[info procs ::apps::${app}::start]" != ""} {
                        ::apps::${app}::start
                        append msg "\n$app (re)started"
                    }
                }
            }
        } result]} {
            set msg "Error: $result"
        }
        return $msg
        #}}}
    }; # proc reload 

    ##  Feste Apps (eher nicht)
    #TODO sattelbildanzeige (als Beispiel?)

    ## Apps aus custom/apps laden und initialisieren
    # (Anwendungs-)Module laden
    set app_loaded ""; # Das muß das Modul beim Laden setzen.
    foreach tclfile [glob $::dir/custom/apps/*.tcl] {
        if {[catch {source $tclfile} load_error]} {
            srvLog [namespace current] Error "$load_error\n$errorInfo"
            continue
        }
        if {"$app_loaded" == ""} {
            srvLog {::kernel} Error "app_loaded nicht gesetzt von $tclfile"
        } else {
            dict set apps_loaded $app_loaded $tclfile
            srvLog {::kernel} Notice "$app_loaded geladen aus $tclfile"
        }
    }
    # Module initialisieren
    foreach app_loaded [dict keys $apps_loaded] {
        if {"[info procs ${app_loaded}::init]" != ""} {
            if {[catch {${app_loaded}::init} init_error]} {
                srvLog [namespace current] Error $init_error
            } else {
                srvLog [namespace current] Info "$app_loaded initialisiert"
            }
        } else {
            srvLog [namespace current] Warn "${app_loaded}::init existiert nicht. App wird nicht initialisiert."
        }
    }

}; # ::apps


# Deamon erfolgreich gestartet
# Falls es einen Link auf eine Fallback-Version gibt, wird der nicht mehr gebraucht.
if {[file exists /usr/local/vmkstationd_last]} {
    file delete /usr/local/vmkstationd_last
    srvLog {} Notice "/usr/local/vmkstationd_last gelöscht."
}

vwait forever
exec stty echo

