#
# custom/kernel/jpegsattel.tcl
#
# Sattel-Livebild und individuelle Generierung
#

namespace eval JPEGSattel {
    variable S_OVERLOAD_MSGS 5; # Dauer der Überlastzählung
    variable n_total 0; # Anzahl angeforderter Bilder in $S_OVERLOAD_MSGS Sekunden
    variable n_overloads 0; # Anzahl fehlender Bilder in $S_OVERLOAD_MSGS Sekunden
    variable overload_handler_procnames [list]
    # Ablage-Dictionary für die erzeugten JPEG-Daten. Schlüssel ist der Imagetyp.
    variable jpeg_images [dict create blank [druckbild::jpegdata {*}$::JPEGOPTIONS [lrepeat [expr 28*16] 0] 28 16]]
    # Die Callbacks für die Fertigmeldungen.
    # Schlüssel ist der Imagetyp, Wert ist eine Liste der zugehörigen Handler.
    variable image_handlers [dict create]
    variable dbg2counter 0

    # Callback für die Übernahme der Druckwerte von TTYSattel
    # Startet die JPEG-Generierung, sofern es mindestens einen Handler gibt
    # n_rows wie vom Controller geliefert
    # n_cols wie vom Controller geliefert
    # => Zeilen und Spalten werden beim Sattel vertauscht
    proc nextImage {druckbild n_rows n_cols} {; #{{{
        variable dbg2counter
        variable image_handlers

        # Bildgenerierung nur, wenn das evtl. jemand wissen will.
        if {[dict exists $image_handlers sattel_live] > 0} {
            if {$dbg2counter % 30 == 0} {
                # Das wird noch gedreht.
                srvLog [namespace current] Debug "::nextImage sattel_live BxH=$n_cols*$n_rows"
            }
            # incr dbg2counter in jpegFinished
            # Zeilen und Spalten werden beim Sattel vertauscht
            ::DBLD2IMG::startJPEG [namespace current]::jpegFinished $druckbild $n_cols $n_rows sattel_live
        }
        #}}}
    }; # proc nextImage 
    
    
    # Callback für die Weitergabe des erzeugten Druckbildes vom Sattel
    # (Aufruf wird aus dem Generierungsthread heraus angeschoben)
    # Holt das Bild, speichert es in jpeg_images und ruft die registrierten Handler auf
    proc jpegFinished {imagetype} {; #{{{
        variable jpeg_images
        variable dbg2counter
        variable image_handlers

        if {[catch {dict set jpeg_images $imagetype [::tsv::get images $imagetype]} msg]} {
            # $imagetype wahrscheinlich falsch
            srvLog [namespace current] Error "::jpegFinished: $msg"
            return
        }
        # Callbacks aufrufen
        set n_callbacks 0; # DEBUG
        if {[dict exists $image_handlers $imagetype]} {; # Es gibt Handler für $imagetype
            # Handler durchgehen
            foreach image_handler [dict get $image_handlers $imagetype] {
                $image_handler $imagetype
                incr n_callbacks
            }
        }
        # Die Sattelmatte liefert, auch wenn sie nur herumliegt, 12 Bilder je Sekunde.
        # Das wird wird selbst im Loglevel "Debug" zu viel.
        if {$dbg2counter % 30 == 0} {
            srvLog [namespace current]::jpegFinished Debug "$imagetype ($n_callbacks callbacks invoked)"
        }
        incr dbg2counter
        #}}}
    }; # proc jpegFinished


    # Erzeugte Bilder löschen
    # @param args   Auflistung der Bildtypen
    proc unsetImages {args} {; #{{{
        variable jpeg_images

        foreach imagetype $args {
            dict unset jpeg_images $imagetype
            #??? srvLog [namespace current]::unsetImages Debug "Image $imagetype unset [dict exists $::kernel::JPEGSattel::jpeg_images $imagetype]."
        }
        #}}}
    }; # proc unsetImages 


    # Globale JPEG-Optionen ändern
    # @param args   key/value Paare:
    #                   -colorcontrast 0...7
    #                   -bgcolor #%2x%2x%2x
    proc setGlobalJPEG {args} {; #{{{
        foreach {key value} $args {
            ::druckbild::photo_configure $key $value
        }
        #}}}
    }; # proc setGlobalJPEG 


    # Erzeugung eines JPEGs aus einem Druckbild mit alternativen JPEG-Optionen
    # Die erzeugten JPEG-Daten werden von jpegFinished entgegengenommen.
    # @param druckbild          Druckdaten
    # @param n_rows             Anzahl Zeilen
    # @param n_cols             Anzahl Spalten
    # @param alt_jpegoptions    abweichende JPEG-Options (Liste)
    #   -mask 1|0               Maske anzeigen ja/nein
    #   -resolution <ppp>       Pixel pro Meßpunkt
    #   -grid no|solid|dotted   Gitter
    #   -frame <pixels>         Rahmenstärke
    #   -quality <prozent>      JPEG-Qualität
    #   -maximum <max>          Maximalwert (falls nicht der tatsächliche)
    proc createJPEG {imagetype druckbild n_rows n_cols {alt_jpegoptions {}} } {; #{{{
        variable n_total

        ::DBLD2IMG::startJPEG [namespace current]::jpegFinished $druckbild $n_rows $n_cols $imagetype $alt_jpegoptions
        incr n_total
        #}}}
    }; # proc createJPEG 


    # Weiteren Handler hinzufügen
    # Der Handler muß den Bildtyp entgegennehmen.
    # @param cb_image_handler   Name der Callbackprozedur
    # @param imagetypes         Liste der Bildtypen, die der Handler bedient
    proc addImageHandler {cb_image_handler imagetypes} {; #{{{
        variable image_handlers

        if {[llength $imagetypes] == 0} {
            srvLog [namespace current] Error "::addImageHandler: must provide at least 1 imagetype"
            return
        }
        # Vorsichtshalber prüfen, ob es den schon gibt
        foreach {imagetype} $imagetypes {
            if {[dict exists $image_handlers $imagetype]} {; # Es gibt Handler für $imagetype, ...
                if {[lsearch [dict get $image_handlers $imagetype] $cb_image_handler] < 0} {
                    # ... aber nicht diesen.
                    dict lappend image_handlers $imagetype $cb_image_handler
                    srvLog [namespace current]::addImageHandler Debug "$cb_image_handler added for $imagetype"
                } else {
                    srvLog [namespace current]::addImageHandler Debug "$cb_image_handler already added for $imagetype"
                }
            } else {
                dict set image_handlers $imagetype $cb_image_handler
                srvLog [namespace current]::addImageHandler Debug "$cb_image_handler added for $imagetype"
            }
        }
        #}}}
    }; # proc addImageHandler 


    # Handler entfernen
    proc removeImageHandler {cb_image_handler} {; #{{{
        variable image_handlers

        set not_found 1
        dict for {imagetype handlers} $image_handlers {; # Bildtypen durchgehen
            set i [lsearch [dict get $image_handlers $imagetype] $cb_image_handler]
            if {$i >=0} {
                set new_handlers [lreplace [dict get $image_handlers $imagetype] $i $i]
                if {[llength $new_handlers] == 0} {; # kein Handler mehr für diesen Bildtyp
                    dict unset image_handlers $imagetype
                } else {
                    dict set image_handlers $imagetype $new_handlers
                }
                srvLog [namespace current]::removeImageHandler Debug "$cb_image_handler removed for $imagetype"
                set not_found 0
            }
        }; # Bildtypen durchgehen
        if {$not_found} {
            srvLog [namespace current]::removeImageHandler Warn "$cb_image_handler not found"
        }
        #}}}
    }; # proc removeImageHandler 


    # Überlastmeldung an die Handler verteilen
    proc disposeOverloadMsg {} {; #{{{
        variable n_total
        variable n_overloads
        variable overload_handler_procnames

        foreach overload_handler_procname $overload_handler_procnames {
            $overload_handler_procname [list total $n_total rejected $n_overloads]
        }
        set n_overloads 0
        #}}}
    }; # proc disposeOverloadMsg 


    # Handler für Überlastungsmeldung des JPEG-Generators
    proc jpegOverload {imagetype} {; #{{{
        variable S_OVERLOAD_MSGS
        variable n_total
        variable n_overloads
        #TODO overloads nach $imagetype ?

        incr n_overloads
        if {$n_overloads == 1} {
            set n_total 1; # createJPEG zählt weiter.
            after [expr $S_OVERLOAD_MSGS * 1000] [namespace current]::disposeOverloadMsg 
        }
        #}}}
    }; # proc jpegOverload 


    # (Weiteren) Handler für Überlastung hinzufügen
    # Der muß den Bildtyp als Argument entgegennehmen.
    # @param cb_overload_handler    Name der Callback-Prozedur
    proc addOverloadHandler {cb_overload_handler} {; #{{{
        variable overload_handler_procnames

        # Vorsichtshalber prüfen, ob es den schon gibt
        if {[lsearch $overload_handler_procnames $cb_overload_handler] < 0} {
            lappend overload_handler_procnames $cb_overload_handler
            srvLog "[namespace current]::addOverloadHandler" Debug "$cb_overload_handler added"
        }
        #}}}
    }; #proc addOverloadHandler 


    # Handler für Überlastung entfernen
    # @param cb_overload_handler    Name der Callback-Prozedur
    proc removeOverloadHandler {cb_overload_handler} {; #{{{
        variable overload_handler_procnames

        set i [lsearch $overload_handler_procnames $cb_overload_handler]
        if {$i >=0} {
            set overload_handler_procnames [lreplace $overload_handler_procnames $i $i]
            srvLog [namespace current] Debug "::removeOverloadHandler $cb_overload_handler added"
        } else {
            srvLog [namespace current] Warn "::removeOverloadHandler $cb_overload_handler not found"
        }
        #}}}
    }; # proc removeOverloadHandler 


    # Modul-Initialisierung
    proc init {} {
        # Anmelden bei TTYSattel
        #TODO Das ist wohl nur noch wegen einer nicht mehr benötigten Kompatibilität zu Darrens Erstfassung.
        ::kernel::TTYSattel::addImageHandler [namespace current]::nextImage
        # Anmelden bei der JPEG-Generierung
        ::DBLD2IMG::addOverloadHandler [namespace current]::jpegOverload
    }

}; # namespace eval JPEGSattel

set mod_loaded JPEGSattel

